package ru.yandex.direct.core.entity.adgroup.service.complex;

import java.util.List;
import java.util.Map;
import java.util.Set;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.adgroup.container.AdGroupUpdateOperationParams;
import ru.yandex.direct.core.entity.adgroup.container.ComplexAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupSimple;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupsUpdateOperation;
import ru.yandex.direct.core.entity.adgroup.service.ModerationMode;
import ru.yandex.direct.core.entity.banner.service.BannersAddOperation;
import ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseValidator;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.regions.GeoTree;

import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupUpdateConverter.adGroupsToModelChanges;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Операция обновления "большой" группы - древовидной группы,
 * содержащей вложенные объекты: "большие" баннеры и условия показа.
 * "Большие" баннеры в свою очередь так же имеют древовидную структуру
 * и содержат набор сайтлинков и визитку.
 * <p>
 * Модели больших баннеров и групп реализованы таким образом,
 * что содержат саму модель группы, а так же вложеные модели. Фактически
 * получается, что не большая группа наследуется от обычной и дополняет
 * ее вложенными моделями (баннерами и условиями показа), а содержит
 * и группу, и вложенные модели на одном уровне ({@link ComplexAdGroup}.
 * Это сделано по той причине, что наследование в группах используется
 * для реализации различных типов групп, и использовать наследование
 * одновременно для "ортогональной" функциональности невозможно.
 * (!) Однако, результат валидации возвращается по простым группам,
 * но при этом содержит результаты валидации под-моделей, тоже простых
 * как если бы простая группа действительно содержала простые баннеры.
 * Таким образом, {@link ComplexAdGroup} используется лишь как условный
 * контейнер для линковки групп с под-моделями, то же самое касается
 * и баннеров. Кажется, что результат валидации в такой форме будет удобен
 * для отдачи в веб, но это еще предстоит проверить на практике.
 * <p>
 * Операция обновления древовидной группы подобно самой этой группе
 * имеет древовидную структуру. Древовидная стуктура аналогична классу
 * группы, но не ее экземпляру. Она содержит под-операции обновления
 * списка больших баннеров и условий показа, а под-операция обновления
 * списка больших баннеров в свою очередь содержит под-операции
 * обновления визиток и сайтлинков.
 * <p>
 * Операция выполняется следующим образом. При создании операции
 * древовидная структура моделей разворачивается в плоские списки
 * моделей - групп, баннеров, визиток и т.п., с сохранением
 * первоначальной связи по индексам. Операция обновления простых
 * групп ({@link AdGroupsUpdateOperation} создается внутри данного класса.
 * Для обновления плоского списка больших баннеров создается
 * соответствующая под-операция, в которую передается единый плоский
 * список больших баннеров из всех обновляемых групп. Внутри
 * этой операции простые баннеры разделяются на два списка:
 * обновляемые и добавляемые. Для этих списков создаются операции
 * над простыми баннерами:
 * {@link BannersAddOperation}
 * и {@link ru.yandex.direct.core.entity.banner.service.BannersUpdateOperation}.
 * По такому же принципу инициализируется вся древовидная операция.
 * Все операции для работы с простыми объектами создаются со специальным
 * флагом {@code partOfComplexOperation = true}, который указывает им,
 * что данная операция над простыми объектами является частью операции
 * над деревом объектов. В зависимости от этого флага логика выполнения
 * операций может немного меняться, особенно это касается валидации.
 * Когда вызывается метод {@link #prepareAndApply()}, сначала проводится валидация
 * для каждого типа объектов в определенном порядке, а так же
 * в отдельном сервисе валидации, специфичном для обновления дерева,
 * проводятся специфические проверки. Если результат валидации
 * отрицательный, то из результатов валидации отдельных типов
 * моделей собирается древовидный результат валидации, как если
 * бы валидировалось дерево. Если результат положительный,
 * то под-операции выполняются в определенном порядке, обновляя
 * за один запрос все баннеры, все группы и т.д.
 * <p>
 * Всем объектам, вложенным в группу, проставляются id группы,
 * в которую они вложены в запросе, и id кампании, к которой принадлежит
 * эта группа.
 * <p>
 * На уровне группы валидируются права на просмотр/редактирование кампании,
 * к которой принадлежит обновляемая группа, а так же, что кампания не является архивной.
 * <p>
 * На уровне баннеров не валидируются права, так как баннер считается частью группы,
 * права на обновление которой проверяются на уровне группы. Вместе с этим
 * в отдельном сервисе валидации большой группы проверяется, что
 * обновляемые баннеры принадлежат группе, в которую они вложены в запросе.
 * <p>
 * На уровне визитки не проверяются права на кампанию, так как id кампании
 * ей выставляется прямо здесь в соответствии с тем, к какой кампании принадлежит группа,
 * в которую вложена эта визитка. А права на указанную кампанию проверяются на уровне группы.
 */
public class ComplexAdGroupUpdateOperation<T extends ComplexAdGroup> extends AbstractComplexAdGroupOperation<T, AdGroupsUpdateOperation> {
    protected final Map<Long, AdGroupSimple> adGroupSimpleMap;
    protected final ModerationMode moderationMode;
    protected final ru.yandex.direct.core.entity.banner.service.moderation.ModerationMode bannersModerationMode;

    public ComplexAdGroupUpdateOperation(Applicability applicability, ModerationMode moderationMode,
                                         AdGroupService adGroupService, List<T> complexAdGroups, GeoTree geoTree,
                                         long operatorUid, ClientId clientId) {
        super(applicability, complexAdGroups, clientId, adGroupService.createPartialUpdateOperation(
                adGroupsToModelChanges(mapList(complexAdGroups, ComplexAdGroup::getAdGroup)),
                AdGroupUpdateOperationParams.builder()
                        .withModerationMode(moderationMode)
                        .withValidateInterconnections(false)
                        .build(),
                MinusPhraseValidator.ValidationMode.ONE_ERROR_PER_TYPE_AND_KEYWORD, geoTree, operatorUid, clientId));

        this.moderationMode = moderationMode;
        switch (moderationMode) {
            case DEFAULT:
                this.bannersModerationMode =
                        ru.yandex.direct.core.entity.banner.service.moderation.ModerationMode.DEFAULT;
                break;
            case FORCE_SAVE_DRAFT:
                this.bannersModerationMode =
                        ru.yandex.direct.core.entity.banner.service.moderation.ModerationMode.FORCE_SAVE_DRAFT;
                break;
            case FORCE_MODERATE:
                this.bannersModerationMode =
                        ru.yandex.direct.core.entity.banner.service.moderation.ModerationMode.FORCE_MODERATE;
                break;
            default:
                throw new IllegalStateException("unknown moderation mode: " + moderationMode);
        }

        // (!) данные доставать из базы строго по clientId,
        // так как на данный момент еще никакие id не провалидированы!
        Set<Long> affectedAdGroupIdsTmp = StreamEx.of(adGroups).map(AdGroup::getId).toSet();
        this.adGroupSimpleMap = adGroupService.getSimpleAdGroups(clientId, affectedAdGroupIdsTmp);
    }

    public ComplexAdGroupUpdateOperation(Applicability applicability, boolean saveDraft,
                                         AdGroupService adGroupService, List<T> complexAdGroups, GeoTree geoTree,
                                         long operatorUid, ClientId clientId) {
        this(applicability, saveDraft ? ModerationMode.FORCE_SAVE_DRAFT : ModerationMode.FORCE_MODERATE,
                adGroupService, complexAdGroups, geoTree, operatorUid, clientId);
    }
}
