package ru.yandex.direct.core.entity.adgroup.service.complex;

import java.net.URL;
import java.util.Set;
import java.util.function.Predicate;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.banner.container.ComplexBanner;
import ru.yandex.direct.core.entity.banner.model.BannerWithHref;
import ru.yandex.direct.core.entity.banner.model.BannerWithOrganization;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.BannerWithTurboLanding;
import ru.yandex.direct.core.entity.banner.model.BannerWithVcard;
import ru.yandex.direct.core.entity.banner.model.PerformanceBannerMain;
import ru.yandex.direct.core.entity.banner.model.TextBanner;
import ru.yandex.direct.core.entity.sitelink.model.Sitelink;
import ru.yandex.direct.core.entity.sitelink.model.SitelinkSet;
import ru.yandex.direct.core.entity.vcard.model.Vcard;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.adGroupTypeNotSupported;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.invalidSitelinkSetIdUsage;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.requiredButEmptyHrefOrTurboOrVcardIdOrPermalink;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.sitelinkDomainIsNotFitToBanner;
import static ru.yandex.direct.core.entity.banner.service.validation.type.BannerTypeValidationPredicates.isImageBanner;
import static ru.yandex.direct.core.entity.banner.service.validation.type.BannerTypeValidationPredicates.isTextBanner;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isNull;

public class ComplexAdGroupValidationCommons {

    public static Constraint<AdGroupType, Defect> adGroupTypeIsApplicable(Set<AdGroupType> applicableTypes) {
        Predicate<AdGroupType> predicate = applicableTypes::contains;
        return Constraint.fromPredicate(predicate, adGroupTypeNotSupported());
    }

    public static ValidationResult<SitelinkSet, Defect> validateSitelinkSet(ComplexBanner complexBanner,
                                                                            SitelinkSet sitelinkSet) {
        ModelItemValidationBuilder<SitelinkSet> vb = ModelItemValidationBuilder.of(sitelinkSet);

        if (isTextBanner(complexBanner.getBanner())) {
            if (sitelinkSet == null) {
                return vb.getResult();
            }
            var banner = (TextBanner) complexBanner.getBanner();
            String bannerHrefStr = banner.getHref();
            URL bannerHref = bannerHrefStr == null ? null : createUrlIgnoringException(bannerHrefStr);

            vb.list(SitelinkSet.SITELINKS)
                    .checkEachBy(sitelink -> validateSitelink(bannerHref, sitelink), When.isTrue(bannerHref != null));
        }

        if (isImageBanner(complexBanner.getBanner())) {
            vb.check(isNull());
        }

        return vb.getResult();
    }

    private static ValidationResult<Sitelink, Defect> validateSitelink(URL bannerHref, Sitelink sitelink) {
        ModelItemValidationBuilder<Sitelink> vb = ModelItemValidationBuilder.of(sitelink);
        if (sitelink == null) {
            return vb.getResult();
        }

        // Проверка в perl отключена, пока живём без проверки на сервере: https://st.yandex-team.ru/DIRECT-69180
        //vb.item(Sitelink.HREF).check(sitelinkDomainIsFitToBanner(bannerHref));

        return vb.getResult();
    }

    private static Constraint<String, Defect> sitelinkDomainIsFitToBanner(URL bannerHref) {
        Predicate<String> predicate = sitelinkHrefStr -> {
            if (sitelinkHrefStr == null || bannerHref.getHost() == null) {
                return true;
            }

            URL sitelinkHref = createUrlIgnoringException(sitelinkHrefStr);
            //noinspection SimplifiableIfStatement
            if (sitelinkHref == null) {
                return true;
            }

            return bannerHref.getHost().equals(sitelinkHref.getHost());
        };
        return fromPredicate(predicate, sitelinkDomainIsNotFitToBanner());
    }

    public static ValidationResult<Vcard, Defect> validateVcard(ComplexBanner complexBanner,
                                                                Vcard vcard) {
        ModelItemValidationBuilder<Vcard> vb = ModelItemValidationBuilder.of(vcard);

        if (isImageBanner(complexBanner.getBanner())) {
            vb.check(isNull());
        }

        return vb.getResult();
    }

    public static Constraint<BannerWithSystemFields, Defect> hrefOrVcardOrTurboOrPermalinkIsSet(
            ComplexBanner complexBanner,
            Boolean clientHasDesktopTurbolandingFeature) {
        Predicate<BannerWithSystemFields> predicate = banner ->
                complexBanner.getVcard() != null
                        || (banner instanceof PerformanceBannerMain)
                        || (banner instanceof BannerWithHref && ((BannerWithHref) banner).getHref() != null)
                        || (banner instanceof BannerWithVcard && ((BannerWithVcard) banner).getVcardId() != null)
                        || (banner instanceof BannerWithOrganization && ((BannerWithOrganization) banner).getPermalinkId() != null)
                        || isBannerWithoutHrefAllowed(complexBanner, clientHasDesktopTurbolandingFeature);
        return fromPredicate(predicate, requiredButEmptyHrefOrTurboOrVcardIdOrPermalink());
    }

    public static Constraint<BannerWithSystemFields, Defect> sitelinksCanExistOnlyWithBannerHref(
            ComplexBanner complexBanner, Boolean clientHasDesktopTurbolandingFeature) {
        Predicate<BannerWithSystemFields> predicate = banner -> complexBanner.getSitelinkSet() == null
                || (banner instanceof BannerWithHref && ((BannerWithHref) banner).getHref() != null)
                || isBannerWithoutHrefAllowed(complexBanner, clientHasDesktopTurbolandingFeature);
        return fromPredicate(predicate, invalidSitelinkSetIdUsage());
    }

    public static URL createUrlIgnoringException(String url) {
        try {
            return url == null ? null : new URL(url);
        } catch (Exception e) {
            return null;
        }
    }

    private static Boolean isBannerWithoutHrefAllowed(
            ComplexBanner complexBanner, Boolean clientHasDesktopTurbolandingFeature) {
        if (clientHasDesktopTurbolandingFeature) {
            BannerWithTurboLanding b = (BannerWithTurboLanding) complexBanner.getBanner();
            return b.getTurboLandingId() != null;
        } else {
            return false;
        }

    }
}
