package ru.yandex.direct.core.entity.adgroup.service.complex;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import com.google.common.collect.Multimap;
import com.google.common.collect.SetMultimap;
import one.util.streamex.EntryStream;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.container.ComplexAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.banner.container.ComplexBanner;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.service.text.BannerTextExtractor;
import ru.yandex.direct.core.entity.banner.type.language.BannerLanguageValidator;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatch;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.queryrec.QueryrecService;
import ru.yandex.direct.queryrec.model.Language;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupConstraints.adGroupGeoMatchesWithUntouchedBannersLanguage;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupConstraints.modifiedRelevanceMatchCorrespondsToAdGroup;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupConstraints.newBannersCountIsInBound;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupConstraints.updatedBannerCorrespondsToAdGroup;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupConstraints.updatedKeywordCorrespondsToAdGroup;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupValidationCommons.hrefOrVcardOrTurboOrPermalinkIsSet;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupValidationCommons.sitelinksCanExistOnlyWithBannerHref;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupValidationCommons.validateSitelinkSet;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupValidationCommons.validateVcard;
import static ru.yandex.direct.core.entity.banner.service.validation.BannerConstraints.isBannerClassCorrespondTo;

@Service
public class UpdateComplexAdGroupValidationService {

    private final QueryrecService queryrecService;
    private final BannerLanguageValidator bannerLanguageValidator;
    private final BannerTextExtractor bannerTextExtractor;
    private final FeatureService featureService;

    @Autowired
    public UpdateComplexAdGroupValidationService(
            QueryrecService queryrecService,
            BannerLanguageValidator bannerLanguageValidator,
            BannerTextExtractor bannerTextExtractor,
            FeatureService featureService) {
        this.queryrecService = queryrecService;
        this.bannerLanguageValidator = bannerLanguageValidator;
        this.bannerTextExtractor = bannerTextExtractor;
        this.featureService = featureService;
    }

    /**
     * В валидации взаимосвязи группы с баннерами нужно помнить о том,
     * что помимо баннеров в запросе есть еще и существующие баннеры,
     * не затронутые запросом.
     */
    public ValidationResult<AdGroup, Defect> validateAdGroup(
            AdGroup adGroup,
            Map<Long, Language> campaignLanguagesByAdGroupId,
            Map<Long, List<BannerWithSystemFields>> untouchedBannersByAdGroupIds,
            ClientId clientId, Long clientRegionId) {
        Long adGroupId = adGroup.getId();

        Language campaignLanguage = campaignLanguagesByAdGroupId.get(adGroupId);
        var untouchedBanners = untouchedBannersByAdGroupIds.get(adGroupId);

        ModelItemValidationBuilder<AdGroup> vb = ModelItemValidationBuilder.of(adGroup);

        vb.item(AdGroup.GEO)
                .check(adGroupGeoMatchesWithUntouchedBannersLanguage(
                        bannerLanguageValidator, queryrecService, bannerTextExtractor,
                        campaignLanguage, untouchedBanners, clientId, clientRegionId));

        return vb.getResult();
    }

    public ValidationResult<List<BannerWithSystemFields>, Defect> validateComplexBanners(
            AdGroup adGroup, List<ComplexBanner> complexBanners, List<BannerWithSystemFields> banners,
            Map<Long, Language> campaignLanguageByAdGroupId, Multimap<Long, Long> existingBannerIdsByAdGroupIds,
            ClientId clientId, Long clientRegionId
    ) {
        Long adGroupId = adGroup.getId();

        Map<Long, Integer> adGroupBannersCount = EntryStream.of(existingBannerIdsByAdGroupIds.asMap())
                .mapValues(Collection::size)
                .toMap();

        return ListValidationBuilder.<BannerWithSystemFields, Defect>of(banners)
                .check(newBannersCountIsInBound(adGroupId, adGroupBannersCount))
                .checkEach(updatedBannerCorrespondsToAdGroup(adGroupId, existingBannerIdsByAdGroupIds))
                .checkEachBy((index, banner) -> {
                    //noinspection ConstantConditions: в комплексной операции должно проверяться, что список баннеров
                    //не null
                    ComplexBanner complexBanner = complexBanners.get(index);
                    return validateComplexBanner(complexBanner, adGroup, campaignLanguageByAdGroupId,
                            clientId, clientRegionId);
                })
                .getResult();
    }

    public <T extends BannerWithSystemFields> ValidationResult<List<T>, Defect> validateBanners(
            AdGroup adGroup, List<T> banners, Map<Long, Language> campaignLanguageByAdGroupId,
            Multimap<Long, Long> existingBannerIdsByAdGroupIds, ClientId clientId, Long clientRegionId) {
        Long adGroupId = adGroup.getId();

        Map<Long, Integer> adGroupBannersCount = EntryStream.of(existingBannerIdsByAdGroupIds.asMap())
                .mapValues(Collection::size)
                .toMap();

        return ListValidationBuilder.<T, Defect>of(banners)
                .check(newBannersCountIsInBound(adGroupId, adGroupBannersCount))
                .checkEach(updatedBannerCorrespondsToAdGroup(adGroupId, existingBannerIdsByAdGroupIds))
                .checkEachBy(banner -> validateBanner(banner, adGroup,
                        campaignLanguageByAdGroupId, clientId, clientRegionId))
                .getResult();
    }

    /**
     * Валидирует простой баннер ({@link #validateBanner(BannerWithSystemFields, AdGroup, Map, ClientId, Long)})
     * и связанные сущности комплексного баннера
     */
    private ValidationResult<BannerWithSystemFields, Defect> validateComplexBanner(
            ComplexBanner complexBanner,
            AdGroup adGroup,
            Map<Long, Language> campaignLanguageByAdGroupId,
            ClientId clientId,
            Long clientRegionId) {
        Boolean clientHasDesktopTurbolandingFeature =
                featureService.isEnabledForClientId(clientId, FeatureName.DESKTOP_LANDING);
        ModelItemValidationBuilder<BannerWithSystemFields> vb =
                ModelItemValidationBuilder.of(complexBanner.getBanner());

        vb
                .checkBy(banner -> validateBanner(banner, adGroup, campaignLanguageByAdGroupId, clientId,
                        clientRegionId))
                .check(hrefOrVcardOrTurboOrPermalinkIsSet(complexBanner, clientHasDesktopTurbolandingFeature))
                .check(sitelinksCanExistOnlyWithBannerHref(complexBanner, clientHasDesktopTurbolandingFeature));

        vb.item(complexBanner.getSitelinkSet(), ComplexBanner.SITELINK_SET.name())
                .checkBy(sitelinkSet -> validateSitelinkSet(complexBanner, sitelinkSet));

        vb.item(complexBanner.getVcard(), ComplexBanner.VCARD.name())
                .checkBy(vcard -> validateVcard(complexBanner, vcard));

        return vb.getResult();
    }

    private <T extends BannerWithSystemFields> ValidationResult<T, Defect> validateBanner(
            T banner, AdGroup adGroup,
            Map<Long, Language> campaignLanguageMap, ClientId clientId,
            Long clientRegionId) {
        boolean isNewBanner = banner.getId() == null;
        Language campaignLang = campaignLanguageMap.get(adGroup.getId());

        ModelItemValidationBuilder<T> vb = ModelItemValidationBuilder.of(banner);

        vb.check(bannerLanguageValidator.newLanguageIsFromGeo(adGroup.getGeo(), campaignLang, clientId, clientRegionId),
                When.isTrue(adGroup.getId() != null));

        vb.check(isBannerClassCorrespondTo(adGroup.getType()), When.isTrue(isNewBanner));

        return vb.getResult();
    }

    public ValidationResult<List<Keyword>, Defect> validateKeywords(AdGroup adGroup, List<Keyword> keywords,
                                                                    SetMultimap<Long, Long> existingKeywordIdsByAdGroupIds) {
        Long adGroupId = adGroup.getId();

        return ListValidationBuilder.<Keyword, Defect>of(keywords)
                .checkEach(updatedKeywordCorrespondsToAdGroup(adGroupId, existingKeywordIdsByAdGroupIds))
                .getResult();
    }

    public <T extends ComplexAdGroup> ValidationResult<List<RelevanceMatch>, Defect> validateRelevanceMatches(
            T complexAdGroup, List<RelevanceMatch> relevanceMatches,
            Multimap<Long, Long> existingRelevanceIdsByAdGroupIds) {
        Long adGroupId = complexAdGroup.getAdGroup().getId();

        return ListValidationBuilder.<RelevanceMatch, Defect>of(relevanceMatches)
                .checkEach(modifiedRelevanceMatchCorrespondsToAdGroup(adGroupId, existingRelevanceIdsByAdGroupIds))
                .getResult();
    }

}
