package ru.yandex.direct.core.entity.adgroup.service.complex.cpm;

import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.ImmutableSet;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.container.ComplexCpmAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.GeoproductAvailability;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.adgroup.service.complex.AddComplexAdGroupValidationService;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupConstraints.bidModifiersNotLinkedWithoutKeywords;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupConstraints.complexAdGroupHasDemographyBidModifiers;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupConstraints.complexAdGroupHasDeviceBidModifiers;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupConstraints.complexAdGroupHasKeywords;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupConstraints.complexAdGroupHasModifiersWhereKeywordsRequired;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupConstraints.complexAdGroupHasWeatherBidModifiers;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupConstraints.demographyBidModifiersAllowed;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupConstraints.deviceBidModifiersAllowed;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupConstraints.eitherKeywordsOrRetargetingsLinked;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupConstraints.keywordsNotLinked;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupConstraints.weatherBidModifiersAllowed;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupValidationCommons.adGroupTypeIsApplicable;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupConstraints.isAdGroupTypeSupportedWithGeoproduct;

@Service
public class AddComplexCpmAdGroupValidationService {
    private static final Set<AdGroupType> APPLICABLE_TYPES =
            ImmutableSet.of(AdGroupType.CPM_BANNER, AdGroupType.CPM_VIDEO, AdGroupType.CPM_OUTDOOR,
                    AdGroupType.CPM_INDOOR, AdGroupType.CPM_AUDIO, AdGroupType.CPM_GEOPRODUCT,
                    AdGroupType.CPM_GEO_PIN,AdGroupType.CPM_YNDX_FRONTPAGE);

    private final AddComplexAdGroupValidationService addValidationService;
    private final AdGroupService adGroupService;
    private final PriceSalesComplexAdGroupValidationService priceSalesAdGroupValidationService;

    @Autowired
    public AddComplexCpmAdGroupValidationService(
            AddComplexAdGroupValidationService addValidationService,
            AdGroupService adGroupService,
            PriceSalesComplexAdGroupValidationService priceSalesAdGroupValidationService) {
        this.addValidationService = addValidationService;
        this.adGroupService = adGroupService;
        this.priceSalesAdGroupValidationService = priceSalesAdGroupValidationService;
    }

    public ValidationResult<List<AdGroup>, Defect> validateAdGroups(
            ValidationResult<List<AdGroup>, Defect> adGroupsResult,
            List<ComplexCpmAdGroup> complexAdGroups, ClientId clientId) {

        Set<Long> campaignIds = StreamEx.of(complexAdGroups).map(x -> x.getAdGroup().getCampaignId()).toSet();

        Map<Long, CampaignType> campaignTypeMap = priceSalesAdGroupValidationService
                .getCampaignsTypes(clientId, campaignIds);
        Map<Long, GeoproductAvailability> anyGeoproductByCampaignId =
                adGroupService.getGeoproductAvailabilityByCampaignId(clientId, campaignIds);

        new ListValidationBuilder<>(adGroupsResult)
            .checkEachBy((index, adGroup) -> {
                    ComplexCpmAdGroup complexAdGroup = complexAdGroups.get(index);
                    return validateAdGroup(
                            complexAdGroup,
                            anyGeoproductByCampaignId,
                            campaignTypeMap);
                });

        priceSalesAdGroupValidationService
                        .validateAdGroupsByPricePackages(adGroupsResult, complexAdGroups, clientId);

        return adGroupsResult;
    }

    public ValidationResult<AdGroup, Defect> validateAdGroup(
            ComplexCpmAdGroup complexAdGroup,
            Map<Long, GeoproductAvailability> anyGeoproductByCampaignId,
            Map<Long, CampaignType> campaignTypeByCampaignId) {
        AdGroup adGroup = complexAdGroup.getAdGroup();
        ModelItemValidationBuilder<AdGroup> vb = ModelItemValidationBuilder.of(adGroup);

        GeoproductAvailability anyGeoproduct = anyGeoproductByCampaignId.get(adGroup.getCampaignId());
        vb.item(AdGroup.TYPE)
                .check(adGroupTypeIsApplicable(APPLICABLE_TYPES))
                .check(isAdGroupTypeSupportedWithGeoproduct(anyGeoproduct));

        AdGroupType adGroupType = adGroup.getType();
        vb
                .check(eitherKeywordsOrRetargetingsLinked(complexAdGroup),
                        When.isTrue(adGroupType == AdGroupType.CPM_BANNER))
                .check(keywordsNotLinked(complexAdGroup),
                        When.isTrue(adGroupType == AdGroupType.CPM_VIDEO
                                || adGroupType == AdGroupType.CPM_GEOPRODUCT
                                || adGroupType == AdGroupType.CPM_GEO_PIN
                                || adGroupType == AdGroupType.CPM_OUTDOOR
                                || adGroupType == AdGroupType.CPM_INDOOR
                                || adGroupType == AdGroupType.CPM_AUDIO
                                || adGroupType == AdGroupType.CPM_YNDX_FRONTPAGE))
                .check(bidModifiersNotLinkedWithoutKeywords(complexAdGroup),
                        When.isTrue(complexAdGroupHasModifiersWhereKeywordsRequired(complexAdGroup)))
                .check(deviceBidModifiersAllowed(),
                        When.isTrue(complexAdGroupHasDeviceBidModifiers(complexAdGroup) &&
                                // Для CPM_PRICE кампаний не проверям deviceBidModifiersAllowed
                                campaignTypeByCampaignId.get(complexAdGroup.getAdGroup().getCampaignId()) !=
                                        CampaignType.CPM_PRICE))
                .check(weatherBidModifiersAllowed(),
                        When.isTrue(complexAdGroupHasWeatherBidModifiers(complexAdGroup)))
                .check(demographyBidModifiersAllowed(),
                        When.isTrue(complexAdGroupHasDemographyBidModifiers(complexAdGroup)
                                && !complexAdGroupHasKeywords(complexAdGroup)));

        vb.list(complexAdGroup.getBanners(), ComplexCpmAdGroup.BANNERS.name())
                .checkBy(banners -> addValidationService.validateBanners(adGroup, banners));

        return vb.getResult();
    }
}
