package ru.yandex.direct.core.entity.adgroup.service.complex.cpm;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.container.ComplexCpmAdGroup;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktop;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktopAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierMobile;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierMobileAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifier.OsType;
import ru.yandex.direct.core.entity.bidmodifiers.container.ComplexBidModifierConverter;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CpmPriceCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.pricepackage.repository.PricePackageRepository;
import ru.yandex.direct.dbutil.sharding.ShardHelper;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.core.entity.bidmodifier.OsType.ANDROID;
import static ru.yandex.direct.core.entity.bidmodifier.OsType.IOS;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.setUnion;

@Service
public class PriceBidModifierPlatformChooserImpl implements PriceBidModifierPlatformChooser {
    public final ShardHelper shardHelper;
    private final CampaignRepository campaignRepository;
    private final CampaignTypedRepository campaignTypedRepository;
    private final PricePackageRepository pricePackageRepository;
    private final AdGroupRepository adGroupRepository;

    @Autowired
    public PriceBidModifierPlatformChooserImpl(ShardHelper shardHelper, CampaignRepository campaignRepository,
                                               CampaignTypedRepository campaignTypedRepository,
                                               PricePackageRepository pricePackageRepository,
                                               AdGroupRepository adGroupRepository) {
        this.shardHelper = shardHelper;
        this.campaignRepository = campaignRepository;
        this.campaignTypedRepository = campaignTypedRepository;
        this.pricePackageRepository = pricePackageRepository;
        this.adGroupRepository = adGroupRepository;
    }

    void validate(Set<PlatformState> platformsEnabled) {
        long cnt =
                List.of(PlatformState.ALL_MOBILE, PlatformState.ANDROID, PlatformState.IOS)
                        .stream().filter(platformsEnabled::contains).count();

        if (cnt > 1) {
            throw new IllegalStateException("Only one mobile option is available");
        }
    }

    public static Map<Boolean, Set<PlatformState>> filterEnabledPlatform(Collection<BidModifier> bidModifiers) {
        return filterEnabledPlatform(bidModifiers, false);
    }
    public static Map<Boolean, Set<PlatformState>> filterEnabledPlatform(Collection<BidModifier> bidModifiers,
            boolean skipZeroPercent) {
        Map<Boolean, Set<PlatformState>> platformsEnabled = new HashMap<>();
        var isFixed = true;
        platformsEnabled.put(isFixed, new HashSet<>());
        platformsEnabled.put(!isFixed, new HashSet<>());
        for (var bm : bidModifiers) {
            switch (bm.getType()) {
                case MOBILE_MULTIPLIER:
                    BidModifierMobile bidModifierMobile = (BidModifierMobile) bm;
                    BidModifierMobileAdjustment mobileAdj = bidModifierMobile.getMobileAdjustment();
                    if (skipZeroPercent && nvl(mobileAdj.getPercent(), 0) <= 0) {
                        break;
                    }
                    var isRequired = nvl(mobileAdj.getIsRequiredInPricePackage(), true);
                    var set = platformsEnabled.get(isRequired);
                    if (mobileAdj.getOsType() == null) {
                        set.add(PlatformState.ALL_MOBILE);
                    } else if (mobileAdj.getOsType() == IOS) {
                        set.add(PlatformState.IOS);
                    } else if (mobileAdj.getOsType() == OsType.ANDROID) {
                        set.add(PlatformState.ANDROID);
                    } else {
                        throw new IllegalStateException("Unknown OS type: " + mobileAdj.getOsType());
                    }
                    break;
                case DESKTOP_MULTIPLIER:
                    BidModifierDesktop bidModifierDesktop = (BidModifierDesktop) bm;
                    BidModifierDesktopAdjustment desktopAdj = bidModifierDesktop.getDesktopAdjustment();
                    if (skipZeroPercent && nvl(desktopAdj.getPercent(), 0) <= 0) {
                        break;
                    }
                    isRequired = nvl(desktopAdj.getIsRequiredInPricePackage(), true);
                    platformsEnabled.get(isRequired).add(PlatformState.DESKTOP);
                    break;
                default:
                    //pass
                    break;
            }
        }

        return platformsEnabled;
    }

    List<BidModifier> choosePlatforms(Set<PlatformState> platformsEnabled, ComplexCpmAdGroup complexCpmAdGroup) {
        if (platformsEnabled.size() == 0) {
            return emptyList();
        }

        List<BidModifier> result = new ArrayList<>();

        if (!platformsEnabled.contains(PlatformState.DESKTOP)) {
            result.add(
                    new BidModifierDesktop()
                            .withType(BidModifierType.DESKTOP_MULTIPLIER)
                            .withAdGroupId(complexCpmAdGroup.getAdGroup().getId())
                            .withCampaignId(complexCpmAdGroup.getAdGroup().getCampaignId())
                            .withDesktopAdjustment(new BidModifierDesktopAdjustment().withPercent(0))
            );
        }

        OsType osType = null;
        boolean hasDisabledMobile = false;

        if (platformsEnabled.contains(PlatformState.ANDROID)) {
            osType = IOS;
            hasDisabledMobile = true;
        } else if (platformsEnabled.contains(PlatformState.IOS)) {
            osType = ANDROID;
            hasDisabledMobile = true;
        } else if (!platformsEnabled.contains(PlatformState.ALL_MOBILE)
                && platformsEnabled.contains(PlatformState.DESKTOP)
        ) {
            hasDisabledMobile = true;
        }

        if (hasDisabledMobile) {
            result.add(
                    new BidModifierMobile()
                            .withType(BidModifierType.MOBILE_MULTIPLIER)
                            .withAdGroupId(complexCpmAdGroup.getAdGroup().getId())
                            .withCampaignId(complexCpmAdGroup.getAdGroup().getCampaignId())
                            .withMobileAdjustment(new BidModifierMobileAdjustment().withOsType(osType).withPercent(0))
            );
        }

        return result;
    }

    Map<Long, PricePackage> getPricePackageByCampaignIds(int shard, List<Long> campaignIds) {
        Map<Long, CpmPriceCampaign> cpmPriceCampaigns =
                (Map<Long, CpmPriceCampaign>) campaignTypedRepository.getTypedCampaignsMap(shard, campaignIds);
        var pricePackageIds = mapList(cpmPriceCampaigns.values(), CpmPriceCampaign::getPricePackageId);
        var pricePackages = pricePackageRepository.getPricePackages(pricePackageIds);
        var result = new HashMap<Long, PricePackage>();
        cpmPriceCampaigns.values().forEach(campaign ->
                result.put(campaign.getId(), pricePackages.get(campaign.getPricePackageId())));
        return result;
    }

    private Map<ComplexCpmAdGroup, Long> complexCpmAdGroupToCid(int shard, List<ComplexCpmAdGroup> complexAdGroups) {
        Map<ComplexCpmAdGroup, Long> result = new HashMap<>();
        Map<ComplexCpmAdGroup, Long> complexCpmAdGroupToAdGroupId = new HashMap<>();

        for (var group : complexAdGroups) {
            if (group.getAdGroup().getCampaignId() != null) {
                result.put(group, group.getAdGroup().getCampaignId());
            } else {
                if (group.getAdGroup().getId() == null) {
                    throw new IllegalStateException("The group without neither cid nor pid");
                }

                complexCpmAdGroupToAdGroupId.put(group, group.getAdGroup().getId());
            }
        }

        if (!complexCpmAdGroupToAdGroupId.isEmpty()) {
            var campsByAdgroupIds = adGroupRepository.getCampaignIdsByAdGroupIds(shard,
                    complexCpmAdGroupToAdGroupId.values());

            for (var group : complexCpmAdGroupToAdGroupId.keySet()) {
                result.put(group, campsByAdgroupIds.get(group.getAdGroup().getId()));
            }

        }

        return result;
    }

    @Override
    public void addComplexModifiersForPriceAdGroups(int shard, List<ComplexCpmAdGroup> complexAdGroups) {
        Map<ComplexCpmAdGroup, Long> adGroupToCid = complexCpmAdGroupToCid(shard, complexAdGroups);

        var campaignWithTypeByAdGroupId = campaignRepository.getCampaignsTypeMap(shard, adGroupToCid.values());

        List<Long> cpmPriceCampaignIds = filterList(campaignWithTypeByAdGroupId.keySet(),
                campaignId -> campaignWithTypeByAdGroupId.get(campaignId) == CampaignType.CPM_PRICE);

        Map<Long, PricePackage> pricePackageByCampaignId = getPricePackageByCampaignIds(shard, cpmPriceCampaignIds);

        for (ComplexCpmAdGroup complexCpmAdGroup : complexAdGroups) {
            var pack = pricePackageByCampaignId.get(adGroupToCid.get(complexCpmAdGroup));
            if (pack == null) {
                continue;
            }
            var complexBidModifier = complexCpmAdGroup.getComplexBidModifier();
            List<BidModifier> adGroupBidModifiers = (complexBidModifier != null)
                    ? filterList(Arrays.asList(
                            complexBidModifier.getDesktopModifier(),
                            complexBidModifier.getMobileModifier()
                    ), Objects::nonNull)
                    : emptyList();

            Map<Boolean, Set<PlatformState>> adGroupPlatforms = filterEnabledPlatform(
                    setUnion(adGroupBidModifiers, pack.getBidModifiers()));

            complexCpmAdGroup.setComplexBidModifier(
                    ComplexBidModifierConverter.convertToComplexBidModifier(
                            choosePlatforms(adGroupPlatforms.get(true), complexCpmAdGroup)));

        }

    }

}
