package ru.yandex.direct.core.entity.adgroup.service.complex.cpm;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.container.ComplexCpmAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupPriceSales;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.crypta.repository.CryptaSegmentRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.pricepackage.service.PricePackageService;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.Rule;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.core.entity.adgroup.service.AdGroupCpmPriceUtils.isDefaultPriority;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupConstraints.adGroupTargetingsAreInPricePackageLimit;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupConstraints.pricePackageAdGroupAudienceSegmentsDisallowed;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupConstraints.pricePackageAdGroupMetricaSegmentsDisallowed;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupConstraints.pricePackageRetargetingsAllowed;
import static ru.yandex.direct.core.entity.adgroup.service.complex.cpm.PriceBidModifierPlatformChooserImpl.filterEnabledPlatform;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.cpmPriceAdGroupNotAllowedMobilePlatformsCombination;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.cpmPriceAdGroupTypeNotAllowed;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.cpmPriceAdGroupUseNotAllowedBidModifiers;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.flatMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.setUnion;

@Service
public class PriceSalesComplexAdGroupValidationService {
    private final FeatureService featureService;
    private final PricePackageService pricePackageService;
    private final AdGroupRepository adGroupRepository;
    private final CampaignRepository campaignRepository;
    private final CryptaSegmentRepository cryptaSegmentRepository;
    private final ShardHelper shardHelper;

    @Autowired
    public PriceSalesComplexAdGroupValidationService(
            FeatureService featureService,
            PricePackageService pricePackageService,
            AdGroupRepository adGroupRepository,
            CampaignTypedRepository campaignTypedRepository,
            CampaignRepository campaignRepository,
            CryptaSegmentRepository cryptaSegmentRepository,
            ShardHelper shardHelper) {
        this.featureService = featureService;
        this.pricePackageService = pricePackageService;
        this.adGroupRepository = adGroupRepository;
        this.campaignRepository = campaignRepository;
        this.cryptaSegmentRepository = cryptaSegmentRepository;
        this.shardHelper = shardHelper;
    }

    private Map<Long, Long> getCampaignIdsFromComplexAdGroups(int shard, List<ComplexCpmAdGroup> complexAdGroups) {
        // Если группа редактируется, то номера кампаний можно взять по номеру группы.
        // В случае добавления, их в базе еще нет, и брать надо из adGroup.getCampaignId (при редактировании это поле
        // пусто)

        var campaignIds = StreamEx.of(complexAdGroups)
                .mapToEntry(complexAdGroup -> complexAdGroup.getAdGroup().getCampaignId())
                .nonNullValues()
                .mapKeys(complexAdGroup -> complexAdGroup.getAdGroup().getId())
                .toMap((a, b) -> a);

        if (campaignIds.isEmpty()) {
            campaignIds = adGroupRepository.getCampaignIdsByAdGroupIds(shard,
                    mapList(complexAdGroups, complexAdGroup -> complexAdGroup.getAdGroup().getId()));
        }

        return campaignIds;
    }

    public ValidationResult<List<AdGroup>, Defect> validateAdGroupsByPricePackages(
            ValidationResult<List<AdGroup>, Defect> adGroupsResult,
            List<ComplexCpmAdGroup> complexAdGroups, ClientId clientId) {
        int shard = shardHelper.getShardByClientId(clientId);

        var campaignIdByAdGroupIds = getCampaignIdsFromComplexAdGroups(shard, complexAdGroups);

        var campaignWithTypeByAdGroupId = campaignRepository.getCampaignsTypeMap(shard,
                campaignIdByAdGroupIds.values());

        List<Long> cpmPriceCampaignIds = filterList(campaignWithTypeByAdGroupId.keySet(),
                campaignId -> campaignWithTypeByAdGroupId.get(campaignId) == CampaignType.CPM_PRICE);

        Set<String> enabledFeatures = featureService.getEnabledForClientId(clientId);

        Map<Long, PricePackage> pricePackageByCampaignId =
                pricePackageService.getPricePackageByCampaignIds(clientId, cpmPriceCampaignIds);

        var complexAdGroupRetargetingGoals = flatMap(complexAdGroups,
                adGroup -> flatMap(
                        flatMap(nvl(filterList(adGroup.getRetargetingConditions(), Objects::nonNull), emptyList()),
                                retCond -> nvl(filterList(retCond.getRules(), Objects::nonNull), emptyList())),
                        Rule::getGoals)
        );

        //Заполняем parentId у целей, чтобы в дальнейшем по ним валидировать.
        var dbGoals = cryptaSegmentRepository.getByIds(mapList(complexAdGroupRetargetingGoals, Goal::getId));
        filterList(complexAdGroupRetargetingGoals, goal -> dbGoals.get(goal.getId()) != null)
                .forEach(goal -> goal.setParentId(
                        (goal.getParentId() == null) ? dbGoals.get(goal.getId()).getParentId() : goal.getParentId()));

        return new ListValidationBuilder<>(adGroupsResult)
                .checkEachBy((index, adGroup) -> {
                    ComplexCpmAdGroup complexAdGroup = complexAdGroups.get(index);
                    Long campaignId = campaignIdByAdGroupIds.get(complexAdGroup.getAdGroup().getId());
                    PricePackage pricePackage = pricePackageByCampaignId.get(campaignId);

                    ValidationResult<AdGroup, Defect> vr = validateAdGroupByPricePackageTargeting(
                            complexAdGroup,
                            enabledFeatures,
                            pricePackage);

                    vr.merge(validateAdGroupByPricePackageBidModifiers(complexAdGroup, pricePackage));
                    return vr;
                })
                .getResult();
    }

    private ValidationResult<AdGroup, Defect> validateAdGroupByPricePackageTargeting(
            ComplexCpmAdGroup complexAdGroup,
            Set<String> enabledFeatures,
            PricePackage pricePackage) {
        AdGroup adGroup = complexAdGroup.getAdGroup();
        ModelItemValidationBuilder<AdGroup> vb = ModelItemValidationBuilder.of(adGroup);

        if (pricePackage == null ||
                pricePackage.getTargetingsCustom().getRetargetingCondition() == null ||
                vb.check(Constraint.fromPredicate(ad -> pricePackage.getAvailableAdGroupTypes().contains(adGroup.getType()),
                        cpmPriceAdGroupTypeNotAllowed())).getResult().hasAnyErrors()) {
            return vb.getResult();
        }

        if (adGroup.getType() == AdGroupType.CPM_YNDX_FRONTPAGE
            || pricePackage.isFrontpageVideoPackage()) {//Видео на морде валидируется по правилам классической морды
            var priority = ((AdGroupPriceSales) adGroup).getPriority();
            var isDefaultAdGroup = isDefaultPriority(priority);
            vb.check(adGroupTargetingsAreInPricePackageLimit(complexAdGroup, pricePackage),
                    When.isValidAnd(When.isTrue(isDefaultAdGroup)));
            return vb.getResult();
        }

        vb
                .check(pricePackageAdGroupMetricaSegmentsDisallowed(complexAdGroup),
                        When.isValidAnd(When.isFalse(pricePackage.getTargetingsCustom().getRetargetingCondition().getAllowMetrikaSegments())))
                .check(pricePackageAdGroupAudienceSegmentsDisallowed(complexAdGroup),
                        When.isValidAnd(When.isFalse(pricePackage.getTargetingsCustom().getRetargetingCondition().getAllowAudienceSegments())))
                .check(pricePackageRetargetingsAllowed(complexAdGroup, pricePackage), When.isValid())
                .check(adGroupTargetingsAreInPricePackageLimit(complexAdGroup, pricePackage), When.isValid());

        return vb.getResult();
    }

    private ValidationResult<AdGroup, Defect> validateAdGroupByPricePackageBidModifiers(
            ComplexCpmAdGroup complexAdGroup,
            PricePackage pricePackage) {

        ModelItemValidationBuilder<AdGroup> vb = ModelItemValidationBuilder.of(complexAdGroup.getAdGroup());

        var complexBidModifier = complexAdGroup.getComplexBidModifier();

        if (pricePackage == null || complexAdGroup.getComplexBidModifier() == null) {
            return vb.getResult();
        }

        var packagePlatforms = filterEnabledPlatform(pricePackage.getBidModifiers());
        Set<PlatformState> packagePlatformsFixed = packagePlatforms.get(true);
        Set<PlatformState> packagePlatformsOpt = packagePlatforms.get(false);

        Set<PlatformState> adGroupPlatforms = filterEnabledPlatform(filterList(Arrays.asList(
                complexBidModifier.getDesktopModifier(),
                complexBidModifier.getMobileModifier()), Objects::nonNull), true)
                .get(true);

        vb
            .check(
                    // В группе должны быть корректировки только из разрешенных на пакете.
                    Constraint.fromPredicate(ad ->
                                    adGroupPlatforms.isEmpty() ||
                                            setUnion(packagePlatformsFixed, packagePlatformsOpt)
                                                    .containsAll(adGroupPlatforms),
                            cpmPriceAdGroupUseNotAllowedBidModifiers()))
            .check(
                    // Мобильные платформы может быть либо "Все", либо Android, либо iOS
                    Constraint.fromPredicate(ad ->
                                List.of(PlatformState.ALL_MOBILE, PlatformState.ANDROID, PlatformState.IOS)
                                        .stream().filter(setUnion(adGroupPlatforms, packagePlatformsFixed)::contains).count() <= 1,
                            cpmPriceAdGroupNotAllowedMobilePlatformsCombination()), When.isValid());

        return vb.getResult();
    }

    public Map<Long, CampaignType> getCampaignsTypes(ClientId clientId, Set<Long> campaignIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return campaignRepository.getCampaignsTypeMap(shard, campaignIds);
    }
}
