package ru.yandex.direct.core.entity.adgroup.service.complex.cpm;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.Multimap;
import com.google.common.collect.SetMultimap;
import com.google.common.collect.Sets;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.container.ComplexCpmAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.service.complex.UpdateComplexAdGroupValidationService;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.repository.ClientRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.keyword.repository.KeywordRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.queryrec.model.Language;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupConstraints.bidModifiersNotLinkedWithoutKeywords;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupConstraints.complexAdGroupHasDemographyBidModifiers;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupConstraints.complexAdGroupHasDeviceBidModifiers;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupConstraints.complexAdGroupHasKeywords;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupConstraints.complexAdGroupHasModifiersWhereKeywordsRequired;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupConstraints.complexAdGroupHasWeatherBidModifiers;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupConstraints.demographyBidModifiersAllowed;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupConstraints.deviceBidModifiersAllowed;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupConstraints.eitherKeywordsOrRetargetingsLinked;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupConstraints.keywordsNotLinked;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupConstraints.retargetingsOnlyToDemographics;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupConstraints.weatherBidModifiersAllowed;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class UpdateComplexCpmAdGroupValidationService {

    private final UpdateComplexAdGroupValidationService updateValidationService;
    private final CampaignRepository campaignRepository;
    private final BannerTypedRepository bannerTypedRepository;
    private final BannerRelationsRepository bannerRelationsRepository;
    private final KeywordRepository keywordRepository;
    private final FeatureService featureService;
    private final ShardHelper shardHelper;
    private final ClientRepository clientRepository;
    private final PriceSalesComplexAdGroupValidationService priceSalesAdGroupValidationService;

    @Autowired
    public UpdateComplexCpmAdGroupValidationService(
            UpdateComplexAdGroupValidationService updateValidationService,
            CampaignRepository campaignRepository,
            BannerTypedRepository bannerTypedRepository,
            BannerRelationsRepository bannerRelationsRepository,
            KeywordRepository keywordRepository,
            FeatureService featureService,
            ShardHelper shardHelper,
            ClientRepository clientRepository,
            PriceSalesComplexAdGroupValidationService priceSalesAdGroupValidationService) {
        this.updateValidationService = updateValidationService;
        this.campaignRepository = campaignRepository;
        this.bannerTypedRepository = bannerTypedRepository;
        this.bannerRelationsRepository = bannerRelationsRepository;
        this.keywordRepository = keywordRepository;
        this.featureService = featureService;
        this.shardHelper = shardHelper;
        this.clientRepository = clientRepository;
        this.priceSalesAdGroupValidationService = priceSalesAdGroupValidationService;
    }

    public ValidationResult<List<AdGroup>, Defect> validateAdGroups(
            ValidationResult<List<AdGroup>, Defect> adGroupsResult, List<ComplexCpmAdGroup> complexAdGroups,
            ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        ValidationData validationData = obtainValidationData(shard, clientId, complexAdGroups);

        Set<String> enabledFeatures = featureService.getEnabledForClientId(clientId);
        Set<Long> campaignIds = StreamEx.of(complexAdGroups).map(x -> x.getAdGroup().getCampaignId()).toSet();
        Long clientRegionId = clientRepository.getCountryRegionIdByClientId(shard, clientId).orElse(null);
        Map<Long, CampaignType> campaignTypeMap = priceSalesAdGroupValidationService
                .getCampaignsTypes(clientId, campaignIds);

        new ListValidationBuilder<>(adGroupsResult)
                .checkEachBy(adGroup -> updateValidationService
                        .validateAdGroup(adGroup, validationData.adGroupIdToCampaignLanguageMap,
                                validationData.untouchedBannersByAdGroupIds, clientId, clientRegionId))
                .checkEachBy((index, adGroup) -> {
                    ComplexCpmAdGroup complexAdGroup = complexAdGroups.get(index);
                    return validateInterconnections(complexAdGroup, validationData, enabledFeatures, clientId,
                            clientRegionId, campaignTypeMap);
                });
        priceSalesAdGroupValidationService
                .validateAdGroupsByPricePackages(adGroupsResult, complexAdGroups, clientId);

        return adGroupsResult;
    }

    private ValidationResult<AdGroup, Defect> validateInterconnections(
            ComplexCpmAdGroup complexAdGroup,
            ValidationData validationData,
            Set<String> enabledFeatures,
            ClientId clientId,
            Long clientRegionId,
            Map<Long, CampaignType> campaignTypeByCampaignId) {
        ModelItemValidationBuilder<AdGroup> vb = ModelItemValidationBuilder.of(complexAdGroup.getAdGroup());

        AdGroupType adGroupType = complexAdGroup.getAdGroup().getType();
        vb
                .check(eitherKeywordsOrRetargetingsLinked(complexAdGroup),
                        When.isTrue(adGroupType == AdGroupType.CPM_BANNER))
                .check(keywordsNotLinked(complexAdGroup),
                        When.isTrue(adGroupType == AdGroupType.CPM_VIDEO
                                || adGroupType == AdGroupType.CPM_GEOPRODUCT
                                || adGroupType == AdGroupType.CPM_GEO_PIN
                                || adGroupType == AdGroupType.CPM_OUTDOOR
                                || adGroupType == AdGroupType.CPM_INDOOR
                                || adGroupType == AdGroupType.CPM_YNDX_FRONTPAGE))
                .check(retargetingsOnlyToDemographics(complexAdGroup),
                        When.isTrue(adGroupType == AdGroupType.CPM_INDOOR))
                .check(bidModifiersNotLinkedWithoutKeywords(complexAdGroup),
                        When.isTrue(complexAdGroupHasModifiersWhereKeywordsRequired(complexAdGroup)))
                .check(deviceBidModifiersAllowed(),
                        When.isTrue(complexAdGroupHasDeviceBidModifiers(complexAdGroup) &&
                                // Для CPM_PRICE кампаний не проверям deviceBidModifiersAllowed
                                campaignTypeByCampaignId.get(complexAdGroup.getAdGroup().getCampaignId()) !=
                                        CampaignType.CPM_PRICE))
                .check(weatherBidModifiersAllowed(),
                        When.isTrue(complexAdGroupHasWeatherBidModifiers(complexAdGroup)))
                .check(demographyBidModifiersAllowed(),
                        When.isTrue(complexAdGroupHasDemographyBidModifiers(complexAdGroup)
                                && !complexAdGroupHasKeywords(complexAdGroup)));

        vb.list(complexAdGroup.getBanners(), ComplexCpmAdGroup.BANNERS.name())
                .checkBy(banners -> updateValidationService.validateBanners(complexAdGroup.getAdGroup(), banners,
                        validationData.adGroupIdToCampaignLanguageMap,
                        validationData.existingBannerIdsByAdGroupIds, clientId, clientRegionId));

        vb.list(complexAdGroup.getKeywords(), ComplexCpmAdGroup.KEYWORDS.name())
                .checkBy(list -> updateValidationService
                        .validateKeywords(complexAdGroup.getAdGroup(), complexAdGroup.getKeywords(),
                                validationData.existingKeywordIdsByAdGroupIds));

        return vb.getResult();
    }

    private ValidationData obtainValidationData(int shard, ClientId clientId, List<ComplexCpmAdGroup> complexAdGroups) {
        List<Long> adGroupIds = mapList(complexAdGroups, complexAdGroup -> complexAdGroup.getAdGroup().getId());

        Map<Long, Language> adGroupIdToCampaignLanguageMap =
                campaignRepository.getCampaignsLangByAdGroupIds(shard, clientId, adGroupIds);

        Multimap<Long, Long> existingBannerIdsByAdGroupIds =
                bannerRelationsRepository.getAdGroupIdToBannerIds(shard, adGroupIds);
        Set<Long> existingBannerIds = new HashSet<>(existingBannerIdsByAdGroupIds.values());
        Set<Long> requestBannerIds = StreamEx.of(complexAdGroups)
                .flatCollection(ComplexCpmAdGroup::getBanners)
                .map(Banner::getId)
                .toSet();
        Set<Long> untouchedBannerIds = Sets.difference(existingBannerIds, requestBannerIds);
        // достаются баннеры целиком, так как не хочется использовать
        // интерфейс BannerWithLangFields (его хочется переделать)
        var untouchedBanners = bannerTypedRepository.getStrictlyFullyFilled(shard, untouchedBannerIds,
                BannerWithSystemFields.class);
        var untouchedBannersByAdGroupIds = StreamEx.of(untouchedBanners)
                .groupingBy(BannerWithSystemFields::getAdGroupId, Collectors.toList());

        SetMultimap<Long, Long> existingKeywordIdsByAdGroupIds =
                keywordRepository.getKeywordIdsByAdGroupIds(shard, clientId, adGroupIds);

        return new ValidationData(adGroupIdToCampaignLanguageMap, existingBannerIdsByAdGroupIds,
                untouchedBannersByAdGroupIds, existingKeywordIdsByAdGroupIds);
    }

    private static class ValidationData {
        private final Map<Long, Language> adGroupIdToCampaignLanguageMap;
        private final Multimap<Long, Long> existingBannerIdsByAdGroupIds;
        private final Map<Long, List<BannerWithSystemFields>> untouchedBannersByAdGroupIds;
        private final SetMultimap<Long, Long> existingKeywordIdsByAdGroupIds;

        ValidationData(Map<Long, Language> adGroupIdToCampaignLanguageMap,
                       Multimap<Long, Long> existingBannerIdsByAdGroupIds,
                       Map<Long, List<BannerWithSystemFields>> untouchedBannersByAdGroupIds,
                       SetMultimap<Long, Long> existingKeywordIdsByAdGroupIds) {
            this.adGroupIdToCampaignLanguageMap = adGroupIdToCampaignLanguageMap;
            this.existingBannerIdsByAdGroupIds = existingBannerIdsByAdGroupIds;
            this.untouchedBannersByAdGroupIds = untouchedBannersByAdGroupIds;
            this.existingKeywordIdsByAdGroupIds = existingKeywordIdsByAdGroupIds;
        }
    }
}
