package ru.yandex.direct.core.entity.adgroup.service.complex.dynamic;

import java.util.List;

import ru.yandex.direct.core.entity.adgroup.container.ComplexDynamicAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupAddOperation;
import ru.yandex.direct.core.entity.adgroup.service.complex.suboperation.add.BannerLogicSupplier;
import ru.yandex.direct.core.entity.adgroup.service.complex.suboperation.add.BidModifiersLogicSupplier;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.service.BannersAddOperationFactory;
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierService;
import ru.yandex.direct.core.entity.bidmodifiers.service.ComplexBidModifierService;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.singleton;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupValidationCommons.adGroupTypeIsApplicable;

public class ComplexDynamicAdGroupAddOperation extends ComplexAdGroupAddOperation<ComplexDynamicAdGroup> {

    private final BannerLogicSupplier<ComplexDynamicAdGroup, BannerWithSystemFields> bannersLogicSupplier;
    private final BidModifiersLogicSupplier<ComplexDynamicAdGroup> bidModifiersLogicSupplier;

    public ComplexDynamicAdGroupAddOperation(boolean saveDraft,
                                             AdGroupService adGroupService,
                                             BidModifierService bidModifierService,
                                             ComplexBidModifierService complexBidModifierService,
                                             CampaignRepository campaignRepository,
                                             List<ComplexDynamicAdGroup> complexAdGroups,
                                             GeoTree geoTree,
                                             long operatorUid, ClientId clientId, int shard,
                                             BannersAddOperationFactory bannersAddOperationFactory) {
        super(Applicability.FULL, saveDraft, adGroupService, complexAdGroups, geoTree, operatorUid, clientId, true);

        bannersLogicSupplier = new BannerLogicSupplier<>(complexAdGroups, ComplexDynamicAdGroup.BANNERS,
                bannersAddOperationFactory, operatorUid, clientId, saveDraft);
        bidModifiersLogicSupplier =
                new BidModifiersLogicSupplier<>(complexAdGroups, ComplexDynamicAdGroup.COMPLEX_BID_MODIFIER,
                        campaignRepository, bidModifierService, complexBidModifierService, operatorUid, clientId,
                        shard);
    }

    @Override
    protected void afterAdGroupsPrepare(ValidationResult<List<AdGroup>, Defect> adGroupsResult) {
        validateInterconnections(adGroupsResult);

        bannersLogicSupplier.prepare(adGroupsResult);
        bidModifiersLogicSupplier.prepare(adGroupsResult);
    }

    @Override
    protected void afterAdGroupsApply(MassResult<Long> addAdGroupsResult) {
        bannersLogicSupplier.apply(addAdGroupsResult);
        bidModifiersLogicSupplier.apply(addAdGroupsResult);
    }

    private void validateInterconnections(ValidationResult<List<AdGroup>, Defect> adGroupsResult) {
        var complexAdGroupsResult = new ListValidationBuilder<>(adGroupsResult)
                .checkEachBy(this::validateAdGroup)
                .getResult();
        checkState(!complexAdGroupsResult.hasErrors() && !complexAdGroupsResult.hasWarnings(),
                "AddComplexAdGroupValidationService must not return operational errors or warnings");
    }

    private ValidationResult<AdGroup, Defect> validateAdGroup(AdGroup adGroup) {
        var vb = ModelItemValidationBuilder.of(adGroup);
        vb.item(AdGroup.TYPE)
                .check(adGroupTypeIsApplicable(singleton(AdGroupType.DYNAMIC)));
        return vb.getResult();
    }
}
