package ru.yandex.direct.core.entity.adgroup.service.complex.dynamic;

import java.util.List;
import java.util.Map;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.container.ComplexDynamicAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.service.complex.UpdateComplexAdGroupValidationService;
import ru.yandex.direct.core.entity.banner.model.BannerWithAdGroupId;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.repository.ClientRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.queryrec.model.Language;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class UpdateComplexDynamicAdGroupValidationService {

    private final UpdateComplexAdGroupValidationService updateValidationService;
    private final CampaignRepository campaignRepository;
    private final BannerTypedRepository typedRepository;
    private final BannerRelationsRepository relationsRepository;
    private final ShardHelper shardHelper;
    private final ClientRepository clientRepository;

    @Autowired
    public UpdateComplexDynamicAdGroupValidationService(
            UpdateComplexAdGroupValidationService updateValidationService,
            CampaignRepository campaignRepository,
            BannerTypedRepository typedRepository,
            BannerRelationsRepository relationsRepository,
            ShardHelper shardHelper,
            ClientRepository clientRepository) {
        this.updateValidationService = updateValidationService;
        this.campaignRepository = campaignRepository;
        this.typedRepository = typedRepository;
        this.relationsRepository = relationsRepository;
        this.shardHelper = shardHelper;
        this.clientRepository = clientRepository;
    }

    public ValidationResult<List<AdGroup>, Defect> validateAdGroups(
            ValidationResult<List<AdGroup>, Defect> adGroupsResult,
            List<ComplexDynamicAdGroup> complexAdGroups, ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        ValidationData validationData = obtainValidationData(shard, clientId, complexAdGroups);
        Long clientRegionId = clientRepository.getCountryRegionIdByClientId(shard, clientId).orElse(null);

        return new ListValidationBuilder<>(adGroupsResult)
                .checkEachBy(adGroup -> updateValidationService
                        .validateAdGroup(adGroup, validationData.campaignLanguageMapByAdGroupId,
                                validationData.untouchedBannersByAdGroupIds, clientId, clientRegionId))
                .getResult();
    }

    private ValidationData obtainValidationData(int shard, ClientId clientId,
                                                List<ComplexDynamicAdGroup> complexAdGroups) {
        List<Long> adGroupIds = mapList(complexAdGroups, complexAdGroup -> complexAdGroup.getAdGroup().getId());
        Map<Long, Language> campaignLanguageMapByAdGroupId =
                campaignRepository.getCampaignsLangByAdGroupIds(shard, clientId, adGroupIds);
        List<BannerWithSystemFields> banners =
                typedRepository.getBannersByGroupIds(shard, adGroupIds, BannerWithSystemFields.class);
        Map<Long, List<BannerWithSystemFields>> bannersByAdGroupId =
                StreamEx.of(banners).groupingBy(BannerWithAdGroupId::getAdGroupId, toList());
        return new ValidationData(campaignLanguageMapByAdGroupId, bannersByAdGroupId);
    }

    private static class ValidationData {
        private final Map<Long, Language> campaignLanguageMapByAdGroupId;
        private final Map<Long, List<BannerWithSystemFields>> untouchedBannersByAdGroupIds;

        ValidationData(Map<Long, Language> campaignLanguageMapByAdGroupId,
                       Map<Long, List<BannerWithSystemFields>> untouchedBannersByAdGroupIds) {
            this.campaignLanguageMapByAdGroupId = campaignLanguageMapByAdGroupId;
            this.untouchedBannersByAdGroupIds = untouchedBannersByAdGroupIds;
        }
    }

}
