package ru.yandex.direct.core.entity.adgroup.service.complex.internal;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.adgroup.container.InternalAdGroupAddItem;
import ru.yandex.direct.core.entity.adgroup.container.InternalAdGroupOperationContainer;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.AdGroupAdditionalTargetingUtils;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.service.AdGroupAdditionalTargetingsAddOperationFactory;
import ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseValidator;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingConditionBase;
import ru.yandex.direct.core.entity.retargeting.model.TargetInterest;
import ru.yandex.direct.core.entity.retargeting.service.AddTargetInterestService;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionOperationFactory;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.operation.AddedModelId;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.PartiallyApplicableOperation;
import ru.yandex.direct.operation.creator.GroupOperationCreator;
import ru.yandex.direct.operation.creator.OperationCreator;
import ru.yandex.direct.operation.creator.ValidationOperationCreator;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static ru.yandex.direct.core.entity.adgroup.service.complex.internal.InternalAdGroupsUtils.convertRetargetingCondition;
import static ru.yandex.direct.core.entity.adgroup.service.complex.internal.InternalAdGroupsUtils.createTargetInterest;
import static ru.yandex.direct.operation.Applicability.isFull;
import static ru.yandex.direct.operation.aggregator.PartiallyApplicableOperationAggregator.mergeMassResults;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Комплексная операция обновления групп внутренней рекламы вместе с дополнительными таргетингами.
 */
@ParametersAreNonnullByDefault
public class NotVeryComplexInternalAdGroupAddOperation implements PartiallyApplicableOperation<Long> {
    private final AddTargetInterestService addTargetInterestService;

    private boolean prepared;
    private boolean executed;
    private MassResult<Long> result;
    private final Applicability applicability;

    private final int shard;
    private final Currency clientCurrency;
    private final long operatorUid;
    private final UidAndClientId owner;
    private final List<InternalAdGroupAddItem> addItems;

    private PartiallyApplicableOperation<Long> addAdGroupsOp;
    private PartiallyApplicableOperation<List<Long>> addAdditionalTargetingOp;
    private PartiallyApplicableOperation<List<AddedModelId>> findOrCreateRetargetingConditionsOp;

    NotVeryComplexInternalAdGroupAddOperation(
            AdGroupService adGroupService,
            AdGroupAdditionalTargetingsAddOperationFactory adGroupAdditionalTargetingsAddOperationFactory,
            RetargetingConditionOperationFactory retargetingConditionOperationFactory,
            AddTargetInterestService addTargetInterestService,
            InternalAdGroupOperationContainer operationContainer, int shard, Currency clientCurrency,
            GeoTree geoTree,
            List<InternalAdGroupAddItem> addItems) {
        this.addTargetInterestService = addTargetInterestService;
        this.applicability = operationContainer.getApplicability();
        this.shard = shard;
        this.operatorUid = operationContainer.getOperatorUid();
        this.owner = operationContainer.getOwner();
        this.clientCurrency = clientCurrency;
        this.addItems = addItems;

        checkArgument(addItems.stream()
                .map(InternalAdGroupAddItem::getAdGroup)
                .allMatch(Objects::nonNull));
        checkArgument(addItems.stream()
                .map(InternalAdGroupAddItem::getAdditionalTargetings)
                .allMatch(Objects::nonNull));
        checkArgument(addItems.stream()
                .map(InternalAdGroupAddItem::getRetargetingConditions)
                .allMatch(Objects::nonNull));


        OperationCreator<AdGroup, PartiallyApplicableOperation<Long>> addAdGroupsOpCreator =
                adGroups -> adGroupService.createAddOperation(
                        adGroups, geoTree, MinusPhraseValidator.ValidationMode.ONE_ERROR_PER_TYPE,
                        operatorUid, owner.getClientId(), Applicability.PARTIAL, operationContainer.isSaveDraft(),
                        true);

        var addAdditionalTargetingsGroupOpCreator = new GroupOperationCreator<>(
                (List<AdGroupAdditionalTargeting> targetingGroups) ->
                        adGroupAdditionalTargetingsAddOperationFactory.newInstance(
                                Applicability.PARTIAL, true, targetingGroups, owner.getClientId()));

        var addAdditionalTargetingsOpCreator = new ValidationOperationCreator<>(addAdditionalTargetingsGroupOpCreator,
                adGroupAdditionalTargetingsAddOperationFactory.getValidatorSupplier().getGroupValidator());

        var findOrCreateRetargetingConditionsOpCreator = new GroupOperationCreator<>(
                (List<RetargetingCondition> retargetingConditions) ->
                        retargetingConditionOperationFactory.createFindOrCreateOperation(
                                Applicability.PARTIAL,
                                retargetingConditions,
                                owner.getClientId(),
                                true)); // для внутренней рекламы не проверяем доступность счётчиков

        // создаём все вложенные операции
        List<AdGroup> adGroups = mapList(addItems, InternalAdGroupAddItem::getAdGroup);
        addAdGroupsOp = addAdGroupsOpCreator.create(adGroups);

        List<List<AdGroupAdditionalTargeting>> additionalTargetingGroups = makeAdditionalTargetingGroupsFromAddItems();
        addAdditionalTargetingOp = addAdditionalTargetingsOpCreator.create(additionalTargetingGroups);

        List<List<RetargetingCondition>> retargetingConditionGroups = mapList(addItems, this::getRetargetingConditions);
        findOrCreateRetargetingConditionsOp =
                findOrCreateRetargetingConditionsOpCreator.create(retargetingConditionGroups);
    }

    public Set<Integer> getValidElementIndexes() {
        checkState(prepared, "prepare() must be called before getValidElementIndexes()");
        return StreamEx.of(
                        addAdGroupsOp.getValidElementIndexes(),
                        addAdditionalTargetingOp.getValidElementIndexes(),
                        findOrCreateRetargetingConditionsOp.getValidElementIndexes())
                .reduce(Sets::intersection)
                .orElseThrow();
    }

    public Optional<MassResult<Long>> prepare() {
        checkState(!prepared, "prepare() can be called only once");
        prepared = true;

        addAdGroupsOp.prepare();
        addAdditionalTargetingOp.prepare();
        findOrCreateRetargetingConditionsOp.prepare();

        Set<Integer> validIndexes = getValidElementIndexes();
        if ((isFull(applicability) && validIndexes.size() != addItems.size()) || validIndexes.isEmpty()) {
            result = cancel();
            return Optional.of(result);
        }
        return Optional.empty();
    }

    public MassResult<Long> cancel() {
        checkApplyOrCancelPreconditions();
        executed = true;

        StreamEx.of(addAdGroupsOp, addAdditionalTargetingOp, findOrCreateRetargetingConditionsOp)
                .filter(op -> op.getResult().isEmpty())
                .forEach(PartiallyApplicableOperation::cancel);

        result = mergeMassResults(
                addItems,
                InternalAdGroupAddItem.AD_GROUP.name(), addAdGroupsOp,
                Map.of(
                        InternalAdGroupAddItem.ADDITIONAL_TARGETINGS.name(), addAdditionalTargetingOp,
                        InternalAdGroupAddItem.RETARGETING_CONDITIONS.name(), findOrCreateRetargetingConditionsOp
                )
        );
        return result;
    }

    public MassResult<Long> apply(Set<Integer> elementIndexesToApply) {
        checkApplyOrCancelPreconditions();
        checkArgument(Sets.difference(elementIndexesToApply, getValidElementIndexes()).isEmpty(),
                "elementIndexesToApply contains indexes of invalid elements");
        executed = true;

        MassResult<Long> resultForAdGroups = addAdGroupsOp.apply(elementIndexesToApply);

        List<List<AdGroupAdditionalTargeting>> targetingGroups = makeAdditionalTargetingGroupsFromAddItems();
        // заполняем adGroupId в таргетингах
        EntryStream.zip(resultForAdGroups.getResult(), targetingGroups)
                .filterKeys(Result::isSuccessful)
                .mapKeys(Result::getResult)
                .flatMapValues(Collection::stream)
                .forKeyValue((adGroupId, targeting) -> targeting.setAdGroupId(adGroupId));

        // применяем таргетинги
        addAdditionalTargetingOp.apply(elementIndexesToApply);

        MassResult<List<AddedModelId>> resultForRetConds =
                findOrCreateRetargetingConditionsOp.apply(elementIndexesToApply);

        // Создаём ретаргетинги
        List<TargetInterest> targetInterests = createTargetInterests(resultForAdGroups.getResult(),
                resultForRetConds.getResult());
        addTargetInterestService.addValidTargetInterests(shard, operatorUid, owner, clientCurrency, targetInterests);

        this.result = mergeMassResults(
                addItems,
                InternalAdGroupAddItem.AD_GROUP.name(), addAdGroupsOp,
                Map.of(
                        InternalAdGroupAddItem.ADDITIONAL_TARGETINGS.name(), addAdditionalTargetingOp,
                        InternalAdGroupAddItem.RETARGETING_CONDITIONS.name(), findOrCreateRetargetingConditionsOp
                )
        );
        return this.result;
    }

    public MassResult<Long> apply() {
        return apply(getValidElementIndexes());
    }

    public Optional<MassResult<Long>> getResult() {
        return Optional.ofNullable(result);
    }

    private List<List<AdGroupAdditionalTargeting>> makeAdditionalTargetingGroupsFromAddItems() {
        return mapList(addItems, addItem -> filterList(addItem.getAdditionalTargetings(),
                AdGroupAdditionalTargetingUtils::isTargetingRelevantForStore));
    }

    private void checkApplyOrCancelPreconditions() {
        checkState(prepared, "prepare() must be called before apply() or cancel()");
        checkState(!executed, "apply() or cancel() can be called only once");
        checkState(result == null, "result is already computed by prepare()");
    }

    private List<RetargetingCondition> getRetargetingConditions(InternalAdGroupAddItem item) {
        LocalDateTime now = LocalDateTime.now();
        List<RetargetingConditionBase> retargetingConditions = nvl(item.getRetargetingConditions(), emptyList());
        return mapList(retargetingConditions, rcBase -> convertRetargetingCondition(owner.getClientId(), rcBase, now));
    }

    private List<TargetInterest> createTargetInterests(List<Result<Long>> adGroupsResultList,
                                                       List<Result<List<AddedModelId>>> retCondsResult) {
        LocalDateTime now = LocalDateTime.now();
        // кампании проставляются во входящих группах и валидируются при добавлении групп
        Map<Long, Long> campaignByAdGroup = StreamEx.of(addItems)
                .map(InternalAdGroupAddItem::getAdGroup)
                .mapToEntry(AdGroup::getId, AdGroup::getCampaignId)
                .nonNullKeys()
                .nonNullValues()
                .toMap();

        return EntryStream.zip(adGroupsResultList, retCondsResult)
                .filterKeys(Result::isSuccessful)
                .mapKeys(Result::getResult)
                .mapValues(Result::getResult)
                .flatMapValues(Collection::stream)
                .mapKeyValue((adGroupId, rcAddInfo) -> {
                    Long campaignId = checkNotNull(campaignByAdGroup.get(adGroupId));
                    Long retargetingConditionId = rcAddInfo.getId();
                    return createTargetInterest(campaignId, adGroupId, retargetingConditionId, now);
                })
                .toList();
    }
}
