package ru.yandex.direct.core.entity.adgroup.service.complex.internal;

import java.time.LocalDateTime;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.direct.core.entity.adgroup.container.AdGroupUpdateOperationParams;
import ru.yandex.direct.core.entity.adgroup.container.InternalAdGroupAddItem;
import ru.yandex.direct.core.entity.adgroup.container.InternalAdGroupOperationContainer;
import ru.yandex.direct.core.entity.adgroup.container.InternalAdGroupUpdateItem;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupWithType;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.adgroup.service.ModerationMode;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.AdGroupAdditionalTargetingUtils;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.service.AdGroupAdditionalTargetingGroupsUpdateOperationFactory;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.service.container.AdGroupAdditionalTargetingGroupsOperationContainer;
import ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseValidator;
import ru.yandex.direct.core.entity.retargeting.model.Retargeting;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingConditionBase;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingDeleteRequest;
import ru.yandex.direct.core.entity.retargeting.model.TargetInterest;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingRepository;
import ru.yandex.direct.core.entity.retargeting.service.AddTargetInterestService;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionOperationFactory;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingDeleteService;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingUtils;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.AddedModelId;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.PartiallyApplicableOperation;
import ru.yandex.direct.operation.creator.GroupOperationCreator;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptySet;
import static ru.yandex.direct.core.entity.adgroup.service.complex.internal.InternalAdGroupsUtils.convertRetargetingCondition;
import static ru.yandex.direct.core.entity.adgroup.service.complex.internal.InternalAdGroupsUtils.createTargetInterest;
import static ru.yandex.direct.operation.Applicability.isFull;
import static ru.yandex.direct.operation.aggregator.PartiallyApplicableOperationAggregator.mergeMassResults;
import static ru.yandex.direct.operation.creator.OperationCreators.createEmptyPartiallyApplicableOperationOnEmptyInput;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.batchDispatch;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapAndFilterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Комплексная операция добавления групп внутренней рекламы вместе с дополнительными таргетингами.
 */
@ParametersAreNonnullByDefault
public class NotVeryComplexInternalAdGroupUpdateOperation implements PartiallyApplicableOperation<Long> {
    private final AdGroupService adGroupService;
    private final AddTargetInterestService addTargetInterestService;
    private final RetargetingRepository retargetingRepository;
    private final RetargetingDeleteService retargetingDeleteService;

    private boolean prepared;
    private boolean executed;
    private MassResult<Long> result;
    private final Applicability applicability;

    private final int shard;
    private final long operatorUid;
    private final UidAndClientId owner;
    private final Currency clientCurrency;
    private final List<InternalAdGroupUpdateItem> updateItems;

    private final PartiallyApplicableOperation<Long> updateAdGroupsOp;
    private final PartiallyApplicableOperation<List<AddedModelId>> updateAdditionalTargetingGroupsOp;
    private final PartiallyApplicableOperation<List<AddedModelId>> findOrCreateRetargetingConditionsOp;
    private final List<Boolean> updateRetargeting;

    NotVeryComplexInternalAdGroupUpdateOperation(
            ShardHelper shardHelper, AdGroupService adGroupService,
            AdGroupAdditionalTargetingGroupsUpdateOperationFactory additionalTargetingGroupsUpdateOperationFactory,
            RetargetingConditionOperationFactory retargetingConditionOperationFactory,
            RetargetingRepository retargetingRepository, AddTargetInterestService addTargetInterestService,
            RetargetingDeleteService retargetingDeleteService,
            InternalAdGroupOperationContainer operationContainer, Currency clientCurrency,
            GeoTree geoTree,
            List<InternalAdGroupUpdateItem> updateItems) {
        this.adGroupService = adGroupService;
        this.addTargetInterestService = addTargetInterestService;
        this.retargetingRepository = retargetingRepository;
        this.retargetingDeleteService = retargetingDeleteService;
        this.applicability = operationContainer.getApplicability();
        this.operatorUid = operationContainer.getOperatorUid();
        this.owner = operationContainer.getOwner();
        this.clientCurrency = clientCurrency;
        this.updateItems = updateItems;

        checkArgument(updateItems.stream()
                .map(InternalAdGroupUpdateItem::getAdGroupChanges)
                .allMatch(Objects::nonNull));
        checkArgument(updateItems.stream()
                .map(InternalAdGroupUpdateItem::getAdditionalTargetings)
                .allMatch(Objects::nonNull));

        // Пока в excel'е не поддержаны таргетинги на основе ретаргетингов, делаем возможность не менять
        // ретаргетинги на группе. Для этого нужно записать null, в качестве списка условий ретаргетингов на группу
        updateRetargeting = updateItems.stream()
                .map(InternalAdGroupUpdateItem::getRetargetingConditions)
                .map(Objects::nonNull)
                .collect(Collectors.toList());

        shard = shardHelper.getShardByClientIdStrictly(owner.getClientId());

        AdGroupUpdateOperationParams operationParams = AdGroupUpdateOperationParams.builder()
                .withModerationMode(operationContainer.isSaveDraft()
                        ? ModerationMode.FORCE_SAVE_DRAFT
                        : ModerationMode.FORCE_MODERATE)
                .withValidateInterconnections(true)
                .build();

        var updateAdGroupOperationCreator = createEmptyPartiallyApplicableOperationOnEmptyInput(
                (List<ModelChanges<AdGroup>> adGroupChanges) -> adGroupService.createUpdateOperation(
                        adGroupChanges, operationParams, geoTree,
                        MinusPhraseValidator.ValidationMode.ONE_ERROR_PER_TYPE, operatorUid, owner.getClientId(),
                        Applicability.PARTIAL));

        var findOrCreateRetargetingConditionsOpCreator = new GroupOperationCreator<>(
                (List<RetargetingCondition> retargetingConditions) ->
                        retargetingConditionOperationFactory.createFindOrCreateOperation(
                                Applicability.PARTIAL, retargetingConditions, owner.getClientId(), true));

        var adGroupChanges = mapList(this.updateItems, this::getAdGroupChanges);
        updateAdGroupsOp = updateAdGroupOperationCreator.create(adGroupChanges);

        var adGroupIds = mapList(adGroupChanges, ModelChanges::getId);
        var additionalTargetingGroups = mapList(this.updateItems, this::getAdditionalTargetings);

        var additionalTargetingContainer =
                new AdGroupAdditionalTargetingGroupsOperationContainer(
                        owner.getClientId(), additionalTargetingGroups, adGroupIds,
                        operationContainer.getRequestSource());

        updateAdditionalTargetingGroupsOp =
                additionalTargetingGroupsUpdateOperationFactory.newInstance(additionalTargetingContainer);

        List<List<RetargetingCondition>> retargetingConditionGroups = mapList(
                this.updateItems, this::getRetargetingConditions);
        findOrCreateRetargetingConditionsOp =
                findOrCreateRetargetingConditionsOpCreator.create(retargetingConditionGroups);
    }

    public Set<Integer> getValidElementIndexes() {
        checkState(prepared, "prepare() must be called before getValidElementIndexes()");
        return StreamEx.of(
                        updateAdGroupsOp.getValidElementIndexes(),
                        updateAdditionalTargetingGroupsOp.getValidElementIndexes(),
                        findOrCreateRetargetingConditionsOp.getValidElementIndexes())
                .reduce(Sets::intersection)
                .orElseThrow();
    }

    public Optional<MassResult<Long>> prepare() {
        checkState(!prepared, "prepare() can be called only once");
        prepared = true;

        updateAdGroupsOp.prepare();
        updateAdditionalTargetingGroupsOp.prepare();
        findOrCreateRetargetingConditionsOp.prepare();

        Set<Integer> validIndexes = getValidElementIndexes();
        if ((isFull(applicability) && validIndexes.size() != updateItems.size()) || validIndexes.isEmpty()) {
            result = cancel();
            return Optional.of(result);
        }
        return Optional.empty();
    }

    public MassResult<Long> cancel() {
        checkApplyOrCancelPreconditions();
        executed = true;

        StreamEx.of(updateAdGroupsOp, updateAdditionalTargetingGroupsOp, findOrCreateRetargetingConditionsOp)
                .filter(op -> op.getResult().isEmpty())
                .forEach(PartiallyApplicableOperation::cancel);

        result = mergeMassResults(
                updateItems,
                InternalAdGroupUpdateItem.AD_GROUP_CHANGES.name(), updateAdGroupsOp,
                Map.of(
                        InternalAdGroupAddItem.ADDITIONAL_TARGETINGS.name(), updateAdditionalTargetingGroupsOp,
                        InternalAdGroupAddItem.RETARGETING_CONDITIONS.name(), findOrCreateRetargetingConditionsOp
                )
        );
        return result;
    }

    public MassResult<Long> apply(Set<Integer> elementIndexesToApply) {
        checkApplyOrCancelPreconditions();
        checkArgument(Sets.difference(elementIndexesToApply, getValidElementIndexes()).isEmpty(),
                "elementIndexesToApply contains indexes of invalid elements");
        executed = true;

        MassResult<Long> resultForAdGroups = updateAdGroupsOp.apply(elementIndexesToApply);

        // todo zakhar: понять как это обобщить:
        // Если у всех групп с таргетингами они не валидные, то результат уже будет вычислен
        if (updateAdditionalTargetingGroupsOp.getResult().isEmpty()) {
            updateAdditionalTargetingGroupsOp.apply(elementIndexesToApply);
        }

        MassResult<List<AddedModelId>> resultForRetConds =
                findOrCreateRetargetingConditionsOp.apply(elementIndexesToApply);

        // Здесь мы должны создать ретаретинги для новых в рамках группы rc, удалить те, которые не упомянуты
        // и не трогать те, которые уже соответствуют запросу на обновление
        createOrDeleteRetargetings(resultForAdGroups.getResult(), resultForRetConds.getResult());

        result = mergeMassResults(
                updateItems,
                InternalAdGroupUpdateItem.AD_GROUP_CHANGES.name(), updateAdGroupsOp,
                Map.of(
                        InternalAdGroupAddItem.ADDITIONAL_TARGETINGS.name(), updateAdditionalTargetingGroupsOp,
                        InternalAdGroupAddItem.RETARGETING_CONDITIONS.name(), findOrCreateRetargetingConditionsOp
                )
        );
        return result;
    }

    public MassResult<Long> apply() {
        return apply(getValidElementIndexes());
    }

    public Optional<MassResult<Long>> getResult() {
        return Optional.ofNullable(result);
    }

    private void checkApplyOrCancelPreconditions() {
        checkState(prepared, "prepare() must be called before apply() or cancel()");
        checkState(!executed, "apply() or cancel() can be called only once");
        checkState(result == null, "result is already computed by prepare()");
    }

    private ModelChanges<AdGroup> getAdGroupChanges(InternalAdGroupUpdateItem item) {
        return item.getAdGroupChanges().castModelUp(AdGroup.class);
    }

    private List<AdGroupAdditionalTargeting> getAdditionalTargetings(InternalAdGroupUpdateItem item) {
        Long adGroupId = item.getAdGroupChanges().getId();
        return mapAndFilterList(item.getAdditionalTargetings(),
                t -> {
                    t.setAdGroupId(adGroupId);
                    return t;
                }, AdGroupAdditionalTargetingUtils::isTargetingRelevantForStore);
    }

    private List<RetargetingCondition> getRetargetingConditions(InternalAdGroupUpdateItem item) {
        LocalDateTime now = LocalDateTime.now();
        List<RetargetingConditionBase> retargetingConditions = nvl(item.getRetargetingConditions(), List.of());
        return mapList(retargetingConditions, rcBase -> convertRetargetingCondition(owner.getClientId(), rcBase, now));
    }

    private void createOrDeleteRetargetings(List<Result<Long>> adGroupsResultList,
                                            List<Result<List<AddedModelId>>> retCondsResult) {
        // Маскируем результаты, которые не должны влиять на ретаргетинги на группе
        List<Optional<Set<Long>>> maskedRetConds = EntryStream.zip(retCondsResult, updateRetargeting)
                .mapKeys(r -> r.isSuccessful() ? Optional.of(listToSet(r.getResult(), AddedModelId::getId)) :
                        Optional.<Set<Long>>empty())
                .mapKeyValue((mayBeRcIdSet, updRetargeting) -> updRetargeting ? mayBeRcIdSet :
                        Optional.<Set<Long>>empty())
                .toList();

        // ID группы -> желаемый набор ID условий ретаргетинга на группе
        Map<Long, Set<Long>> desiredState = EntryStream.zip(adGroupsResultList, maskedRetConds)
                .filterKeys(Result::isSuccessful)
                .mapKeys(Result::getResult)
                .filterValues(Optional::isPresent)
                .mapValues(Optional::get)
                .toMap();

        // т.к. мы берём только успешные результаты, а операция добавления групп содержит валидацию
        // возможности работать с группами оператору, то все ID групп, на этом этапе, доступны
        List<Retargeting> existentRetargetings = retargetingRepository.getRetargetingsByAdGroups(
                shard, desiredState.keySet());

        // ID группы -> существующий набор ID условий ретаргетинга на группе
        Map<Long, Set<Long>> existentState = StreamEx.of(existentRetargetings)
                .mapToEntry(Retargeting::getAdGroupId, Function.identity())
                .mapValues(Retargeting::getRetargetingConditionId)
                .groupingTo(HashSet::new);

        Map<Long, AdGroupWithType> adGroupsWithType = adGroupService.getAdGroupsWithType(
                owner.getClientId(), desiredState.keySet());
        checkState(
                desiredState.keySet().equals(adGroupsWithType.keySet()),
                "Information of each group must be present");
        Map<Long, Long> campaignByAdGroup = EntryStream.of(adGroupsWithType)
                .mapValues(AdGroupWithType::getCampaignId)
                .toMap();
        // Определяем какие связи групп и условий ретаргетинга нужно установить или разрушить, чтобы прийти
        // к желаемому состоянию
        List<RetargetingAction> retargetingActions = StreamEx.of(desiredState.keySet())
                .flatMap(adGroupId -> {
                    var existent = existentState.getOrDefault(adGroupId, emptySet());
                    var desired = checkNotNull(desiredState.get(adGroupId));
                    return Stream.concat(
                            StreamEx.of(Sets.difference(desired, existent))
                                    .map(retCondId -> RetargetingAction.link(adGroupId, retCondId)),
                            StreamEx.of(Sets.difference(existent, desired))
                                    .map(retCondId -> RetargetingAction.unlink(adGroupId, retCondId))
                    );
                })
                .toList();

        batchDispatch(
                retargetingActions, RetargetingAction::isLinkAction,
                toLink -> linkRetargetingConditions(toLink, campaignByAdGroup),
                unlinkActions -> unlinkRetargetingConditions(unlinkActions, existentRetargetings));
    }

    private List<Long> linkRetargetingConditions(List<RetargetingAction> toLink, Map<Long, Long> campaignByAdGroup) {
        LocalDateTime now = LocalDateTime.now();
        List<TargetInterest> targetInterestsToAdd = mapList(
                toLink, ra -> createTargetInterest(
                        checkNotNull(campaignByAdGroup.get(ra.getAdGroupId())),
                        ra.getAdGroupId(), ra.getRetargetingConditionId(), now));
        return addTargetInterestService.addValidTargetInterests(
                shard, operatorUid, owner, clientCurrency, targetInterestsToAdd);
    }

    private List<Long> unlinkRetargetingConditions(List<RetargetingAction> toUnlink,
                                                   List<Retargeting> existentRetargetings) {
        Set<Pair<Long, Long>> toUnlinkSet = listToSet(
                toUnlink, ra -> Pair.of(ra.getAdGroupId(), ra.getRetargetingConditionId()));
        List<RetargetingDeleteRequest> retargetingDeleteRequests = StreamEx.of(existentRetargetings)
                .filter(r -> toUnlinkSet.contains(Pair.of(r.getAdGroupId(), r.getRetargetingConditionId())))
                .map(RetargetingUtils::convertToRetargetingDeleteRequest)
                .toList();
        retargetingDeleteService.delete(shard, owner, operatorUid, retargetingDeleteRequests);
        return mapList(retargetingDeleteRequests, RetargetingDeleteRequest::getRetargetingId);
    }

    private static class RetargetingAction {
        RetargetingConditionAction action;
        Long adGroupId;
        Long retargetingConditionId;

        RetargetingAction(RetargetingConditionAction action, Long adGroupId, Long retargetingConditionId) {
            this.action = action;
            this.adGroupId = adGroupId;
            this.retargetingConditionId = retargetingConditionId;
        }

        static RetargetingAction link(Long adGroupId, Long retargetingConditionId) {
            return new RetargetingAction(RetargetingConditionAction.LINK, adGroupId, retargetingConditionId);
        }

        static RetargetingAction unlink(Long adGroupId, Long retargetingConditionId) {
            return new RetargetingAction(RetargetingConditionAction.UNLINK, adGroupId, retargetingConditionId);
        }

        boolean isLinkAction() {
            return action == RetargetingConditionAction.LINK;
        }

        Long getAdGroupId() {
            return adGroupId;
        }

        Long getRetargetingConditionId() {
            return retargetingConditionId;
        }

        enum RetargetingConditionAction {
            LINK,
            UNLINK
        }
    }
}
