package ru.yandex.direct.core.entity.adgroup.service.complex.mcbanner;

import java.util.List;
import java.util.Map;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.container.ComplexAdGroup;
import ru.yandex.direct.core.entity.adgroup.container.ComplexMcBannerAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.service.complex.AddComplexAdGroupValidationService;
import ru.yandex.direct.core.entity.banner.container.ComplexBanner;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.repository.ClientRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.queryrec.model.Language;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.Collections.singleton;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupValidationCommons.adGroupTypeIsApplicable;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class AddComplexMcBannerAdGroupValidationService {

    private final AddComplexAdGroupValidationService addValidationService;
    private final CampaignRepository campaignRepository;
    private final ShardHelper shardHelper;
    private final ClientRepository clientRepository;

    @Autowired
    public AddComplexMcBannerAdGroupValidationService(AddComplexAdGroupValidationService addValidationService,
                                                      CampaignRepository campaignRepository, ShardHelper shardHelper,
                                                      ClientRepository clientRepository) {
        this.addValidationService = addValidationService;
        this.campaignRepository = campaignRepository;
        this.shardHelper = shardHelper;
        this.clientRepository = clientRepository;
    }

    public ValidationResult<List<AdGroup>, Defect> validateAdGroups(
            ValidationResult<List<AdGroup>, Defect> adGroupsResult, List<ComplexMcBannerAdGroup> complexAdGroups,
            ClientId clientId) {
        List<AdGroup> adGroups = mapList(complexAdGroups, ComplexAdGroup::getAdGroup);
        List<Long> campaignIds = mapList(adGroups, AdGroup::getCampaignId);

        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        Map<Long, Language> campaignLanguageMap = campaignRepository.getCampaignsLang(shard, clientId, campaignIds);

        Long clientRegionId = clientRepository.getCountryRegionIdByClientId(shard, clientId).orElse(null);

        return new ListValidationBuilder<>(adGroupsResult)
                .checkEachBy(this::validateAdGroup)
                .checkEachBy((index, adGroup) -> {
                    ComplexMcBannerAdGroup complexAdGroup = complexAdGroups.get(index);
                    return validateInterconnections(complexAdGroup, campaignLanguageMap, clientId,
                            clientRegionId);
                })
                .getResult();
    }

    private ValidationResult<AdGroup, Defect> validateAdGroup(AdGroup adGroup) {
        ModelItemValidationBuilder<AdGroup> vb = ModelItemValidationBuilder.of(adGroup);

        vb.item(AdGroup.TYPE)
                .check(adGroupTypeIsApplicable(singleton(AdGroupType.MCBANNER)));

        return vb.getResult();
    }

    private ValidationResult<AdGroup, Defect> validateInterconnections(ComplexMcBannerAdGroup complexAdGroup,
                                                                       Map<Long, Language> campaignLanguageMap,
                                                                       ClientId clientId,
                                                                       Long clientRegionId) {
        List<ComplexBanner> complexBanners = complexAdGroup.getComplexBanners();
        var banners = mapList(complexBanners, ComplexBanner::getBanner);
        AdGroup adGroup = complexAdGroup.getAdGroup();

        ModelItemValidationBuilder<AdGroup> vb = ModelItemValidationBuilder.of(adGroup);

        vb.list(banners, ComplexMcBannerAdGroup.COMPLEX_BANNERS.name())
                .checkBy(list -> addValidationService
                        .validateComplexBanners(adGroup, complexBanners, banners,
                                campaignLanguageMap.get(adGroup.getCampaignId()),
                                clientId, clientRegionId));

        return vb.getResult();
    }
}
