package ru.yandex.direct.core.entity.adgroup.service.complex.mobilecontent;

import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.container.ComplexAdGroup;
import ru.yandex.direct.core.entity.adgroup.container.ComplexMobileContentAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.service.complex.AddComplexAdGroupValidationService;
import ru.yandex.direct.core.entity.banner.container.ComplexBanner;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.repository.ClientRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.queryrec.model.Language;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.Collections.singleton;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupConstraints.complexAdGroupHasDemographyBidModifiers;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupConstraints.complexAdGroupHasMobileBidModifiers;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupConstraints.demographyBidModifiersAllowed;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupConstraints.mobileBidModifiersAllowed;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupValidationCommons.adGroupTypeIsApplicable;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class AddComplexMobileContentAdGroupValidationService {

    private final AddComplexAdGroupValidationService addValidationService;
    private final CampaignRepository campaignRepository;
    private final ShardHelper shardHelper;
    private final ClientRepository clientRepository;

    @Autowired
    public AddComplexMobileContentAdGroupValidationService(AddComplexAdGroupValidationService addValidationService,
                                                           CampaignRepository campaignRepository,
                                                           ShardHelper shardHelper,
                                                           ClientRepository clientRepository) {
        this.addValidationService = addValidationService;
        this.campaignRepository = campaignRepository;
        this.shardHelper = shardHelper;
        this.clientRepository = clientRepository;
    }

    @SuppressWarnings({"checkstyle:OneStatementPerLine", "checkstyle:LineLength", "checkstyle:LeftCurly"})
    public ValidationResult<List<AdGroup>, Defect> validateAdGroups(
            ValidationResult<List<AdGroup>, Defect> adGroupsResult, List<ComplexMobileContentAdGroup> complexAdGroups,
            ClientId clientId) {
        List<AdGroup> adGroups = mapList(complexAdGroups, ComplexAdGroup::getAdGroup);
        List<Long> campaignIds = mapList(adGroups, AdGroup::getCampaignId);

        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        Map<Long, Language> campaignLanguageMap = campaignRepository.getCampaignsLang(shard, clientId, campaignIds);

        var uacCampaignIdToType = campaignRepository.getUniversalCampaignType(shard, campaignIds);

        Long clientRegionId = clientRepository.getCountryRegionIdByClientId(shard, clientId).orElse(null);

        return new ListValidationBuilder<>(adGroupsResult)
                .checkEachBy((index, adGroup) -> {
                    ComplexMobileContentAdGroup complexAdGroup = complexAdGroups.get(index);
                    return validateAdGroup(complexAdGroup);
                })
                .checkEachBy((index, adGroup) -> {
                    ComplexMobileContentAdGroup complexAdGroup = complexAdGroups.get(index);
                    return validateInterconnections(complexAdGroup, campaignLanguageMap, clientId, clientRegionId,
                            adGroup.getCampaignId() == null ? null : uacCampaignIdToType.get(adGroup.getCampaignId()));
                }).getResult();
    }

    private ValidationResult<AdGroup, Defect> validateAdGroup(ComplexMobileContentAdGroup complexAdGroup) {
        ModelItemValidationBuilder<AdGroup> vb = ModelItemValidationBuilder.of(complexAdGroup.getAdGroup());

        vb.item(AdGroup.TYPE)
                .check(adGroupTypeIsApplicable(singleton(AdGroupType.MOBILE_CONTENT)));
        vb
                .check(mobileBidModifiersAllowed(),
                        When.isTrue(complexAdGroupHasMobileBidModifiers(complexAdGroup)))
                .check(demographyBidModifiersAllowed(),
                        When.isTrue(complexAdGroupHasDemographyBidModifiers(complexAdGroup)));

        return vb.getResult();
    }

    private ValidationResult<AdGroup, Defect> validateInterconnections(ComplexMobileContentAdGroup complexAdGroup,
                                                                       Map<Long, Language> campaignLanguageMap,
                                                                       ClientId clientId,
                                                                       @Nullable Long clientRegionId,
                                                                       @Nullable CampaignType uacCampaignType) {
        List<ComplexBanner> complexBanners = complexAdGroup.getComplexBanners();
        var banners = mapList(complexBanners, ComplexBanner::getBanner);
        AdGroup adGroup = complexAdGroup.getAdGroup();

        ModelItemValidationBuilder<AdGroup> vb = ModelItemValidationBuilder.of(adGroup);

        vb.list(banners, ComplexMobileContentAdGroup.COMPLEX_BANNERS.name())
                .checkBy(list -> addValidationService
                        .validateComplexBanners(adGroup, complexBanners, banners,
                                campaignLanguageMap.get(adGroup.getCampaignId()),
                                clientId, clientRegionId, uacCampaignType));

        return vb.getResult();
    }
}
