package ru.yandex.direct.core.entity.adgroup.service.complex.mobilecontent;

import java.util.Collections;
import java.util.List;

import javax.annotation.Nullable;

import ru.yandex.direct.core.entity.adgroup.container.ComplexMobileContentAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupAddOperation;
import ru.yandex.direct.core.entity.adgroup.service.complex.suboperation.add.BidModifiersLogicSupplier;
import ru.yandex.direct.core.entity.adgroup.service.complex.suboperation.add.ComplexBannersLogicSupplier;
import ru.yandex.direct.core.entity.adgroup.service.complex.suboperation.add.KeywordLogicSupplier;
import ru.yandex.direct.core.entity.adgroup.service.complex.suboperation.add.RelevanceMatchLogicSupplier;
import ru.yandex.direct.core.entity.adgroup.service.complex.suboperation.add.RetargetingLogicSupplier;
import ru.yandex.direct.core.entity.banner.container.ComplexBanner;
import ru.yandex.direct.core.entity.banner.service.BannersAddOperationFactory;
import ru.yandex.direct.core.entity.banner.service.DatabaseMode;
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierService;
import ru.yandex.direct.core.entity.bidmodifiers.service.ComplexBidModifierService;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.keyword.service.KeywordOperationFactory;
import ru.yandex.direct.core.entity.relevancematch.service.RelevanceMatchService;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingService;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionAutoPriceParams;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionFixedAutoPrices;
import ru.yandex.direct.core.entity.sitelink.service.SitelinkSetService;
import ru.yandex.direct.core.entity.vcard.service.VcardService;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;

public class ComplexMobileContentAdGroupAddOperation extends ComplexAdGroupAddOperation<ComplexMobileContentAdGroup> {

    private final AddComplexMobileContentAdGroupValidationService addValidationService;
    private final ClientId clientId;

    private ComplexBannersLogicSupplier<ComplexMobileContentAdGroup> bannersLogicSupplier;
    private KeywordLogicSupplier<ComplexMobileContentAdGroup> keywordLogicSupplier;
    private RelevanceMatchLogicSupplier<ComplexMobileContentAdGroup> relevanceMatchLogicSupplier;
    private RetargetingLogicSupplier<ComplexMobileContentAdGroup> retargetingLogicSupplier;
    private BidModifiersLogicSupplier<ComplexMobileContentAdGroup> bidModifiersLogicSupplier;

    /**
     * @param showConditionAutoPrices      включает режим автоматического вычисления недостающих ставок в условиях
     *                                     показов
     * @param showConditionAutoPriceParams параметры для автоматического вычисления недостающих ставок в условиях
     *                                     показов. Должен быть не {@code null},
     *                                     если {@code showConditionAutoPrices == true}
     */
    public ComplexMobileContentAdGroupAddOperation(boolean saveDraft,
                                                   ClientService clientService,
                                                   AdGroupService adGroupService,
                                                   KeywordOperationFactory keywordOperationFactory,
                                                   BannersAddOperationFactory bannersAddOperationFactory,
                                                   VcardService vcardService, SitelinkSetService sitelinkSetService,
                                                   RelevanceMatchService relevanceMatchService,
                                                   RetargetingService retargetingService,
                                                   BidModifierService bidModifierService,
                                                   ComplexBidModifierService complexBidModifierService,
                                                   AddComplexMobileContentAdGroupValidationService addValidationService,
                                                   CampaignRepository campaignRepository,
                                                   List<ComplexMobileContentAdGroup> complexAdGroups, GeoTree geoTree,
                                                   boolean showConditionAutoPrices,
                                                   @Nullable ShowConditionAutoPriceParams showConditionAutoPriceParams,
                                                   long operatorUid, ClientId clientId, long clientUid, int shard) {
        super(Applicability.FULL, saveDraft, adGroupService, complexAdGroups, geoTree, operatorUid, clientId, true);
        this.addValidationService = addValidationService;
        this.clientId = clientId;

        ShowConditionFixedAutoPrices showConditionFixedAutoPrices = null;
        if (showConditionAutoPrices) {
            checkArgument(showConditionAutoPriceParams != null,
                    "showConditionAutoPriceParams must be specified in showConditionAutoPrices mode");
            showConditionFixedAutoPrices = showConditionAutoPriceParams.getFixedAutoPrices();
        }

        Currency clientCurrency = clientService.getWorkCurrency(clientId);

        bannersLogicSupplier = new ComplexBannersLogicSupplier<>(complexAdGroups,
                ComplexMobileContentAdGroup.COMPLEX_BANNERS,
                ComplexBanner.BANNER, bannersAddOperationFactory, vcardService, sitelinkSetService,
                operatorUid, clientId, saveDraft, DatabaseMode.ONLY_MYSQL);
        keywordLogicSupplier = new KeywordLogicSupplier<>(complexAdGroups, ComplexMobileContentAdGroup.KEYWORDS,
                keywordOperationFactory, showConditionAutoPrices, showConditionFixedAutoPrices,
                operatorUid, clientId, clientUid);
        relevanceMatchLogicSupplier =
                new RelevanceMatchLogicSupplier<>(complexAdGroups, ComplexMobileContentAdGroup.RELEVANCE_MATCHES,
                        campaignRepository, relevanceMatchService, clientCurrency,
                        showConditionAutoPrices, showConditionAutoPriceParams, operatorUid, clientId, shard);
        retargetingLogicSupplier = new RetargetingLogicSupplier<>(complexAdGroups, Collections.emptyMap(),
                ComplexMobileContentAdGroup.TARGET_INTERESTS, retargetingService, campaignRepository, false,
                showConditionAutoPrices, showConditionFixedAutoPrices, operatorUid, clientId, clientUid, shard);
        bidModifiersLogicSupplier =
                new BidModifiersLogicSupplier<>(complexAdGroups, ComplexMobileContentAdGroup.COMPLEX_BID_MODIFIER,
                        campaignRepository, bidModifierService, complexBidModifierService, operatorUid, clientId,
                        shard);
    }

    @Override
    protected void afterAdGroupsPrepare(ValidationResult<List<AdGroup>, Defect> adGroupsResult) {
        validateInterconnections(adGroupsResult);

        keywordLogicSupplier.prepare(adGroupsResult);
        bannersLogicSupplier.prepare(adGroupsResult);
        relevanceMatchLogicSupplier.prepare(adGroupsResult);
        retargetingLogicSupplier.prepare(adGroupsResult);
        bidModifiersLogicSupplier.prepare(adGroupsResult);
    }

    private void validateInterconnections(ValidationResult<List<AdGroup>, Defect> adGroupsResult) {
        ValidationResult<List<AdGroup>, Defect> complexAdGroupsResult =
                addValidationService.validateAdGroups(adGroupsResult, complexAdGroups, clientId);
        checkState(!complexAdGroupsResult.hasErrors() && !complexAdGroupsResult.hasWarnings(),
                "AddComplexAdGroupValidationService must not return operational errors or warnings");
    }

    @Override
    protected void afterAdGroupsApply(MassResult<Long> addAdGroupsResult) {
        bannersLogicSupplier.apply(addAdGroupsResult);
        keywordLogicSupplier.apply(addAdGroupsResult);
        relevanceMatchLogicSupplier.apply(addAdGroupsResult);
        retargetingLogicSupplier.apply(addAdGroupsResult);
        bidModifiersLogicSupplier.apply(addAdGroupsResult);
    }
}
