package ru.yandex.direct.core.entity.adgroup.service.complex.performance;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.container.ComplexPerformanceAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.service.complex.UpdateComplexAdGroupValidationService;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.repository.ClientRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.queryrec.model.Language;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.Collections.singleton;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupValidationCommons.adGroupTypeIsApplicable;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class UpdateComplexPerformanceAdGroupValidationService {
    private final UpdateComplexAdGroupValidationService updateValidationService;
    private final CampaignRepository campaignRepository;
    private final BannerTypedRepository bannerTypedRepository;
    private final BannerRelationsRepository bannerRelationsRepository;
    private final ClientRepository clientRepository;
    private final ShardHelper shardHelper;

    @Autowired
    public UpdateComplexPerformanceAdGroupValidationService(
            UpdateComplexAdGroupValidationService updateValidationService,
            CampaignRepository campaignRepository,
            BannerTypedRepository bannerTypedRepository,
            BannerRelationsRepository bannerRelationsRepository,
            ClientRepository clientRepository,
            ShardHelper shardHelper) {
        this.updateValidationService = updateValidationService;
        this.campaignRepository = campaignRepository;
        this.bannerTypedRepository = bannerTypedRepository;
        this.bannerRelationsRepository = bannerRelationsRepository;
        this.clientRepository = clientRepository;
        this.shardHelper = shardHelper;
    }

    public ValidationResult<List<AdGroup>, Defect> validateAdGroups(
            ValidationResult<List<AdGroup>, Defect> adGroupsResult,
            List<ComplexPerformanceAdGroup> complexAdGroups, ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        ValidationData validationData = obtainValidationData(shard, clientId, complexAdGroups);
        Long clientRegionId = clientRepository.getCountryRegionIdByClientId(shard, clientId).orElse(null);

        return new ListValidationBuilder<>(adGroupsResult)
                .checkEachBy(this::validateAdGroup)
                .checkEachBy(adGroup -> updateValidationService
                        .validateAdGroup(adGroup, validationData.campaignLanguageByAdGroupId,
                                validationData.untouchedBannersByAdGroupId, clientId, clientRegionId))
                .checkEachBy((index, adGroup) -> {
                    ComplexPerformanceAdGroup complexAdGroup = complexAdGroups.get(index);
                    return validateInterconnections(complexAdGroup, validationData, clientId, clientRegionId);
                })
                .getResult();
    }

    private ValidationResult<AdGroup, Defect> validateAdGroup(AdGroup adGroup) {
        ModelItemValidationBuilder<AdGroup> vb = ModelItemValidationBuilder.of(adGroup);
        vb.item(AdGroup.TYPE)
                .check(adGroupTypeIsApplicable(singleton(AdGroupType.PERFORMANCE)));
        return vb.getResult();
    }

    private ValidationResult<AdGroup, Defect> validateInterconnections(ComplexPerformanceAdGroup complexAdGroup,
                                                                       ValidationData validationData,
                                                                       ClientId clientId,
                                                                       Long clientRegionId) {
        ModelItemValidationBuilder<AdGroup> vb = ModelItemValidationBuilder.of(complexAdGroup.getAdGroup());
        vb.list(complexAdGroup.getBanners(), ComplexPerformanceAdGroup.BANNERS.name())
                .checkBy(list -> updateValidationService
                        .validateBanners(complexAdGroup.getAdGroup(), complexAdGroup.getBanners(),
                                validationData.campaignLanguageByAdGroupId,
                                validationData.existingBannerIdsByAdGroupId,
                                clientId, clientRegionId));
        return vb.getResult();
    }

    private ValidationData obtainValidationData(int shard, ClientId clientId,
                                                List<ComplexPerformanceAdGroup> complexAdGroups) {
        List<Long> adGroupIds = mapList(complexAdGroups, complexAdGroup -> complexAdGroup.getAdGroup().getId());

        Map<Long, Language> campaignLanguageByAdGroupId =
                campaignRepository.getCampaignsLangByAdGroupIds(shard, clientId, adGroupIds);

        Multimap<Long, Long> existingBannerIdsByAdGroupId =
                bannerRelationsRepository.getAdGroupIdToBannerIds(shard, adGroupIds);
        Set<Long> existingBannerIds = new HashSet<>(existingBannerIdsByAdGroupId.values());
        Set<Long> requestBannerIds = StreamEx.of(complexAdGroups)
                .flatCollection(ComplexPerformanceAdGroup::getBanners)
                .map(Banner::getId)
                .toSet();
        Set<Long> untouchedBannerIds = Sets.difference(existingBannerIds, requestBannerIds);
        List<BannerWithSystemFields> untouchedBanners = bannerTypedRepository.getStrictly(shard, untouchedBannerIds,
                BannerWithSystemFields.class);
        Map<Long, List<BannerWithSystemFields>> untouchedBannersByAdGroupIds = StreamEx.of(untouchedBanners)
                .groupingBy(BannerWithSystemFields::getAdGroupId, Collectors.toList());

        return new ValidationData(campaignLanguageByAdGroupId, existingBannerIdsByAdGroupId,
                untouchedBannersByAdGroupIds);
    }

    private static class ValidationData {
        private final Map<Long, Language> campaignLanguageByAdGroupId;
        private final Multimap<Long, Long> existingBannerIdsByAdGroupId;
        private final Map<Long, List<BannerWithSystemFields>> untouchedBannersByAdGroupId;

        ValidationData(Map<Long, Language> campaignLanguageByAdGroupId,
                       Multimap<Long, Long> existingBannerIdsByAdGroupId,
                       Map<Long, List<BannerWithSystemFields>> untouchedBannersByAdGroupId) {
            this.campaignLanguageByAdGroupId = campaignLanguageByAdGroupId;
            this.existingBannerIdsByAdGroupId = existingBannerIdsByAdGroupId;
            this.untouchedBannersByAdGroupId = untouchedBannersByAdGroupId;
        }
    }
}
