package ru.yandex.direct.core.entity.adgroup.service.complex.suboperation;

import java.util.List;

import one.util.streamex.EntryStream;

import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.service.KeywordUtils;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.PathNode;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.keyword.service.KeywordUtils.KEYWORD_WITH_PARENTHESIS_REGEX;
import static ru.yandex.direct.validation.result.PathHelper.index;
import static ru.yandex.direct.validation.util.ValidationUtils.transferIssuesFromValidationToValidationWithNewValue;

/**
 * Общий код, связанный с разделением ключевых фраз по круглым скобкам в саб-операциях ключевых фраз.
 */
public class KeywordSeparationUtils {

    /**
     * На основе входного списка фраз строит новый список,
     * разбивая исходные фразы с круглыми скобками на несколько фраз.
     * <p>
     * При разбивке обновляемой фразы (с id) с круглыми скобками она
     * превращается в одно обновляемую фразу и добавляемые (у них удаляется id).
     * <p>
     * Если входная фраза содержит ошибку использования круглых скобок,
     * то ее индекс добавляется в специальный список. В методе prepare()
     * для таких фраз будет добавлена ошибка.
     *
     * @param sourceKeywords                           список исходных фраз до разделения
     * @param separatedKeywords                        передается пустой список, который в процессе работы метода заполняется
     *                                                 разделенными по круглым скобкам фразами
     * @param sourceKeywordIndexes                     передается пустой список, который в процессе работы метода заполняется
     *                                                 индексами исходных фраз, чтобы запомнить, какой исходной фразе какие
     *                                                 новые соответствуют; индекс в списке соответствует индексу в результирующем
     *                                                 списке, а значение - это индекс исходного списка.
     * @param sourceKeywordIndexesWithParenthesisError индексы ключевых фраз из исходного списка
     *                                                 с ошибками использования скобок
     */
    public static void separateKeywords(List<Keyword> sourceKeywords, List<Keyword> separatedKeywords,
                                        List<Integer> sourceKeywordIndexes, List<Integer> sourceKeywordIndexesWithParenthesisError) {
        for (int i = 0; i < sourceKeywords.size(); i++) {
            Keyword sourceKeyword = sourceKeywords.get(i);

            if (sourceKeyword.getPhrase() == null ||
                    !sourceKeyword.getPhrase().matches(KEYWORD_WITH_PARENTHESIS_REGEX)) {
                separatedKeywords.add(sourceKeyword);
                sourceKeywordIndexes.add(i);
                continue;
            }

            List<String> phrases = KeywordUtils.splitKeywordWithParenthesis(sourceKeyword.getPhrase());
            if (phrases == null) {
                separatedKeywords.add(sourceKeyword);
                sourceKeywordIndexes.add(i);
                sourceKeywordIndexesWithParenthesisError.add(i);
                continue;
            }

            boolean first = true;
            for (String phrase : phrases) {
                Keyword separatedKeyword = KeywordUtils.cloneKeyword(sourceKeyword).withPhrase(phrase);
                if (!first) {
                    separatedKeyword.setId(null);
                }
                separatedKeywords.add(separatedKeyword);
                sourceKeywordIndexes.add(i);
                first = false;
            }
        }
    }

    /**
     * Переносит результаты валидации списка разделенных ключевых фраз
     * в результат валидации списка исходных фраз, используя для этого индексы
     * соответствия исходного списка и разделенного.
     *
     * @param separatedValidationResult результат валидации списка разделенных ключевых фраз
     * @param sourceValidationResult    результат валидации списка исходных ключевых фраз
     */
    public static void separatedValidationResultToSourceValidationResult(
            ValidationResult<List<Keyword>, Defect> separatedValidationResult,
            ValidationResult<List<Keyword>, Defect> sourceValidationResult,
            List<Integer> sourceKeywordIndexes) {
        List<Keyword> sourceKeywords = sourceValidationResult.getValue();

        EntryStream.of(separatedValidationResult.getSubResults())
                .filterValues(ValidationResult::hasAnyErrors)
                .forKeyValue((pathNode, sepKeywordVr) -> {
                    int sepIndex = ((PathNode.Index) pathNode).getIndex();
                    int sourceIndex = sourceKeywordIndexes.get(sepIndex);
                    Keyword sourceKeyword = sourceKeywords.get(sourceIndex);
                    ValidationResult<?, Defect> srcKeywordVr = sourceValidationResult
                            .getOrCreateSubValidationResult(index(sourceIndex), sourceKeyword);
                    transferIssuesFromValidationToValidationWithNewValue(sepKeywordVr, srcKeywordVr);
                });
    }
}
