package ru.yandex.direct.core.entity.adgroup.service.complex.suboperation;

import java.util.List;

import ru.yandex.direct.core.entity.adgroup.container.ComplexCpmAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupWithUsersSegments;
import ru.yandex.direct.core.entity.adgroup.model.UsersSegment;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.userssegments.service.validation.UsersSegmentConstraints.adGroupTypeSupported;
import static ru.yandex.direct.core.entity.userssegments.service.validation.UsersSegmentConstraints.goalTypeSupportedInAdGroup;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.util.ValidationUtils.transferIssuesFromValidationToValidationWithNewValue;

public class UsersSegmentsValidationUtil {

    public static void validateCpmUsersSegmentsOnUpdate(ValidationResult<List<AdGroup>, Defect> adGroupsResult,
                                                        List<ComplexCpmAdGroup> complexAdGroups) {
        validateList(adGroupsResult, complexAdGroups, true);
    }

    public static void validateCpmUserSegmentsOnInsert(ValidationResult<List<AdGroup>, Defect> adGroupsResult,
                                                       List<ComplexCpmAdGroup> complexAdGroups) {
        validateList(adGroupsResult, complexAdGroups, false);
    }

    private static void validateList(ValidationResult<List<AdGroup>, Defect> adGroupsResult,
                                     List<ComplexCpmAdGroup> complexAdGroups, boolean checkAdGroupIdIsNotNull) {
        ListValidationBuilder<ComplexCpmAdGroup, Defect> lvb = ListValidationBuilder.of(complexAdGroups);
        lvb.checkEachBy(complexAdGroup -> validateComplexAdGroup(
                complexAdGroup, checkAdGroupIdIsNotNull), When.notNull());
        transferIssuesFromValidationToValidationWithNewValue(lvb.getResult(), adGroupsResult);
    }

    private static ValidationResult<ComplexCpmAdGroup, Defect> validateComplexAdGroup(ComplexCpmAdGroup complexAdGroup,
                                                                                      boolean checkAdGroupIdIsNotNull) {
        ModelItemValidationBuilder<ComplexCpmAdGroup> vb = ModelItemValidationBuilder.of(complexAdGroup);
        vb.list(ComplexCpmAdGroup.USERS_SEGMENTS)
                .checkEach(notNull())
                .checkEachBy(segment -> validateUsersSegment(
                                segment, complexAdGroup.getAdGroup().getType(), checkAdGroupIdIsNotNull),
                        When.isValid());
        return vb.getResult();
    }

    public static void validateAdGroupWithUsersSegments(
            ModelItemValidationBuilder<? extends AdGroupWithUsersSegments> vb,
            AdGroupType adGroupType,
            boolean checkAdGroupIdIsNotNull) {
        vb.list(AdGroupWithUsersSegments.USERS_SEGMENTS)
                .checkEachBy(segment -> validateUsersSegment(segment, adGroupType, checkAdGroupIdIsNotNull),
                        When.isValid());
    }

    private static ValidationResult<UsersSegment, Defect> validateUsersSegment(UsersSegment segment,
                                                                               AdGroupType adGroupType,
                                                                               boolean checkAdGroupIdIsNotNull) {
        ModelItemValidationBuilder<UsersSegment> vr = ModelItemValidationBuilder.of(segment);

        vr.item(UsersSegment.TYPE)
                .check(notNull())
                .check(goalTypeSupportedInAdGroup(adGroupType), When.isValid());

        vr.item(UsersSegment.AD_GROUP_ID)
                .check(notNull(), When.isTrue(checkAdGroupIdIsNotNull))
                .check(adGroupTypeSupported(adGroupType), When.isValid());

        return vr.getResult();
    }
}
