package ru.yandex.direct.core.entity.adgroup.service.complex.suboperation.add;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;

import ru.yandex.direct.core.entity.adgroup.service.complex.suboperation.KeywordSeparationUtils;
import ru.yandex.direct.core.entity.keyword.container.AdGroupInfoForKeywordAdd;
import ru.yandex.direct.core.entity.keyword.container.KeywordsAddOperationParams;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.service.KeywordOperationFactory;
import ru.yandex.direct.core.entity.keyword.service.KeywordsAddOperation;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionFixedAutoPrices;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.tree.SubOperation;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.core.entity.keyword.model.Keyword.PHRASE;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.keyphrase.PhraseDefects.incorrectUseOfParenthesis;
import static ru.yandex.direct.operation.tree.TreeOperationUtils.prepareNullableOperationAndGetValidationResult;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.index;

public class AddKeywordsSubOperation implements SubOperation<Keyword> {

    private List<Keyword> sourceKeywords = new ArrayList<>();
    private List<Keyword> separatedKeywords = new ArrayList<>();
    private List<Integer> sourceKeywordIndexesWithParenthesisError = new ArrayList<>();
    private List<Integer> sourceKeywordIndexes = new ArrayList<>();

    private KeywordsAddOperation keywordsAddOperation;

    /**
     * @param keywords        список ключевых слов для добавления
     * @param autoPrices      включает режим {@code autoPrices}, см. коммент к классу {@link KeywordsAddOperation}
     * @param fixedAutoPrices контейнер с фиксированными ставками, которые нужно выставить у фразы, если ставка явно
     *                        не указана, но нужна в текущей стратегии. Могут быть ставки не для всех фраз.
     *                        Должен быть не {@code null}, если {@code autoPrices == true}
     */
    public AddKeywordsSubOperation(KeywordOperationFactory keywordOperationFactory, List<Keyword> keywords,
                                   boolean autoPrices, @Nullable ShowConditionFixedAutoPrices fixedAutoPrices,
                                   long operatorUid, ClientId clientId, long clientUid) {
        checkArgument(keywords != null, "sub operation must not be created with null keywords list");
        sourceKeywords = keywords;

        if (sourceKeywords.isEmpty()) {
            return;
        }

        KeywordSeparationUtils.separateKeywords(sourceKeywords, separatedKeywords,
                sourceKeywordIndexes, sourceKeywordIndexesWithParenthesisError);
        if (hasKeywordSeparationErrors()) {
            return;
        }

        KeywordsAddOperationParams operationParams = KeywordsAddOperationParams.builder()
                .withAdGroupsNonexistentOnPrepare(true)
                .withUnglueEnabled(true)
                .withIgnoreOversize(false)
                .withAutoPrices(autoPrices)
                .withWeakenValidation(false)
                .build();
        this.keywordsAddOperation = keywordOperationFactory
                .createKeywordsAddOperation(Applicability.FULL, operationParams, this.separatedKeywords, null,
                        fixedAutoPrices, operatorUid, clientId, clientUid);
    }

    /**
     * @return true, если присутствуют ошибки разделения фраз по круглым скобкам и операция ядра не будет вызвана
     */
    private boolean hasKeywordSeparationErrors() {
        return !sourceKeywordIndexesWithParenthesisError.isEmpty();
    }

    public void setAdGroupInfoByKeywordIndex(Map<Integer, AdGroupInfoForKeywordAdd> adGroupInfoByKeywordIndex) {
        if (keywordsAddOperation != null) {
            Map<Integer, AdGroupInfoForKeywordAdd> newAdGroupInfo = getMapWithNewIndexes(adGroupInfoByKeywordIndex);
            keywordsAddOperation.setKeywordsAdGroupInfo(newAdGroupInfo);
        }
    }

    /**
     * Выставляет id групп добавляемым ключевым фразам.
     * Вызывается после применения операции над группами,
     * так как после этого у них появляется id.
     * <p>
     * Для операции копирования особым образом инициализирует калькулятор
     * для расчета ставок для копируемых и новых фраз.
     *
     * @param adGroupIds индекс ключевика -> id группы
     */
    public void setAdGroupsIds(Map<Integer, Long> adGroupIds) {
        if (keywordsAddOperation != null) {
            Map<Integer, Long> newAdGroupIds = getMapWithNewIndexes(adGroupIds);
            keywordsAddOperation.setAdGroupsIds(newAdGroupIds);
        }
    }

    /**
     * Преобразовывает мапу с информацией о группах вида "индекс объекта -> информация".
     * Если в изначальных фразах были фразы со скобками, количество фраз, передаваемых в операцию, увеличилось.
     * Так как комплексная операция об этом не знает, надо поменять индексы в мапах с информацией о группах.
     * Индексы меняются с помощью {@link #sourceKeywordIndexes}, заполненного в
     * {@link KeywordSeparationUtils#separateKeywords(List, List, List, List)}.
     *
     * @return мапа с измененными индексами
     */
    private <T> Map<Integer, T> getMapWithNewIndexes(Map<Integer, T> infoByIndex) {
        Map<Integer, T> result;
        if (infoByIndex.size() == separatedKeywords.size()) {
            result = infoByIndex;
        } else {
            result = new HashMap<>();
            for (int i = 0; i < separatedKeywords.size(); i++) {
                Integer oldKeywordIndex = sourceKeywordIndexes.get(i);
                result.put(i, infoByIndex.get(oldKeywordIndex));
            }
        }
        return result;
    }

    @Override
    public ValidationResult<List<Keyword>, Defect> prepare() {
        if (hasKeywordSeparationErrors()) {
            // если присутствуют ошибки использования круглых скобок,
            // то собираем результат валидации вручную, не вызывая операцию
            ValidationResult<List<Keyword>, Defect> vr = new ValidationResult<>(sourceKeywords);
            for (int sourceIndexWithError : sourceKeywordIndexesWithParenthesisError) {
                Keyword keyword = sourceKeywords.get(sourceIndexWithError);
                ValidationResult<Keyword, Defect> keywordVr =
                        vr.getOrCreateSubValidationResult(index(sourceIndexWithError), keyword);
                ValidationResult<String, Defect> phraseVr =
                        keywordVr.getOrCreateSubValidationResult(field(PHRASE), keyword.getPhrase());
                phraseVr.addError(incorrectUseOfParenthesis());
            }
            return vr;
        } else {
            ValidationResult<List<Keyword>, Defect> sourceValidationResult =
                    new ValidationResult<>(sourceKeywords);

            ValidationResult<List<Keyword>, Defect> separatedValidationResult =
                    prepareNullableOperationAndGetValidationResult(keywordsAddOperation, separatedKeywords);

            KeywordSeparationUtils.separatedValidationResultToSourceValidationResult(
                    separatedValidationResult, sourceValidationResult, sourceKeywordIndexes);

            return sourceValidationResult;
        }
    }

    @Override
    public void apply() {
        checkState(!hasKeywordSeparationErrors(),
                "AddKeywordsSubOperation.apply() must not be called when error in keywords parenthesis is found");
        if (keywordsAddOperation != null) {
            keywordsAddOperation.apply();
        }
    }
}
