package ru.yandex.direct.core.entity.adgroup.service.complex.suboperation.add;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;

import ru.yandex.direct.core.entity.adgroup.container.ComplexAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.keyword.container.AdGroupInfoForKeywordAdd;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.service.KeywordOperationFactory;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionFixedAutoPrices;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.operation.tree.ListSubOperationExecutor;
import ru.yandex.direct.operation.tree.SubOperationCreator;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupModelUtils.checkKeywordsConsistency;

public class KeywordLogicSupplier<T extends ComplexAdGroup> extends AddSubEntityLogicSupplier<T> {
    private ListSubOperationExecutor<AdGroup, Keyword, AddKeywordsSubOperation> keywordExecutor;

    public KeywordLogicSupplier(List<T> complexAdGroups, ModelProperty<? super T, List<Keyword>> keywordsProperty,
                                KeywordOperationFactory keywordOperationFactory,
                                boolean autoPrices, @Nullable ShowConditionFixedAutoPrices fixedAutoPrices,
                                long operatorUid, ClientId clientId, long clientUid) {
        super(complexAdGroups);
        checkKeywordsConsistency(complexAdGroups, keywordsProperty);
        createKeywordExecutor(keywordsProperty, keywordOperationFactory,
                autoPrices, fixedAutoPrices, operatorUid, clientId, clientUid);
    }

    /**
     * @param autoPrices      включает режим автоматического вычисления недостающих ставок в ключевых фразах.
     *                        См. {@link ru.yandex.direct.core.entity.keyword.service.KeywordsAddOperation}
     * @param fixedAutoPrices контейнер с фиксированными ставками, которые нужно выставить у фразы, если ставка явно
     *                        не указана, но нужна в текущей стратегии. Могут быть ставки не для всех фраз.
     *                        Должен быть не {@code null}, если {@code autoPrices == true}
     */
    private void createKeywordExecutor(ModelProperty<? super T, List<Keyword>> keywordsProperty,
                                       KeywordOperationFactory keywordOperationFactory,
                                       boolean autoPrices, @Nullable ShowConditionFixedAutoPrices fixedAutoPrices,
                                       long operatorUid, ClientId clientId, long clientUid) {
        SubOperationCreator<Keyword, AddKeywordsSubOperation> subOperationCreator =
                allKeywords -> new AddKeywordsSubOperation(keywordOperationFactory, allKeywords,
                        autoPrices, fixedAutoPrices, operatorUid, clientId, clientUid);

        keywordExecutor = ListSubOperationExecutor.builder()
                .withFakeParents(complexAdGroups)
                .withChildrenProperty(keywordsProperty)
                .createSubOperationBy(subOperationCreator);
    }

    @Override
    public void prepare(ValidationResult<List<AdGroup>, Defect> adGroupsResult) {
        Map<Integer, AdGroupInfoForKeywordAdd> adGroupInfoByKeywordIndex = new HashMap<>();
        keywordExecutor.getIndexMultimap().forEach((adGroupIndex, keywordFlatIndex) -> {
            AdGroupInfoForKeywordAdd adGroupInfo = new AdGroupInfoForKeywordAdd(adGroupIndex,
                    adGroups.get(adGroupIndex).getCampaignId(),
                    adGroups.get(adGroupIndex).getType());
            adGroupInfoByKeywordIndex.put(keywordFlatIndex, adGroupInfo);
        });
        keywordExecutor.getSubOperation().setAdGroupInfoByKeywordIndex(adGroupInfoByKeywordIndex);
        keywordExecutor.prepare(adGroupsResult);
    }

    @Override
    public void apply(MassResult<Long> addAdGroupsResult) {
        Map<Integer, Long> adGroupIdsByKeywordIndex = new HashMap<>();
        keywordExecutor.getIndexMultimap().forEach((adGroupIndex, keywordIndex) ->
                adGroupIdsByKeywordIndex.put(keywordIndex, addAdGroupsResult.get(adGroupIndex).getResult()));
        keywordExecutor.getSubOperation().setAdGroupsIds(adGroupIdsByKeywordIndex);
        keywordExecutor.apply();
    }
}
