package ru.yandex.direct.core.entity.adgroup.service.complex.suboperation.add;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;

import ru.yandex.direct.core.entity.adgroup.container.ComplexAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.relevancematch.container.AdGroupInfoForRelevanceMatchAdd;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatch;
import ru.yandex.direct.core.entity.relevancematch.service.RelevanceMatchService;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionAutoPriceParams;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.operation.tree.ListSubOperationExecutor;
import ru.yandex.direct.operation.tree.SubOperationCreator;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupModelUtils.checkRelevanceMatchesConsistency;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

public class RelevanceMatchLogicSupplier<T extends ComplexAdGroup> extends AddSubEntityLogicSupplier<T> {
    private ListSubOperationExecutor<AdGroup, RelevanceMatch, AddRelevanceMatchesSubOperation> relevanceMatchExecutor;
    private CampaignRepository campaignRepository;
    private ClientId clientId;
    private int shard;

    public RelevanceMatchLogicSupplier(List<T> complexAdGroups,
                                       ModelProperty<? super T, List<RelevanceMatch>> relevanceMatchProperty,
                                       CampaignRepository campaignRepository, RelevanceMatchService relevanceMatchService,
                                       Currency clientCurrency, boolean autoPrices, @Nullable ShowConditionAutoPriceParams autoPriceParams,
                                       long operatorUid, ClientId clientId, int shard) {
        super(complexAdGroups);
        checkRelevanceMatchesConsistency(complexAdGroups, relevanceMatchProperty);
        createRelevanceMatchExecutor(relevanceMatchProperty, relevanceMatchService, clientCurrency, autoPrices,
                autoPriceParams, operatorUid, clientId);
        this.campaignRepository = campaignRepository;
        this.clientId = clientId;
        this.shard = shard;
    }


    /**
     * @param autoPrices      включает режим автоматического вычисления недостающих ставок в бесфразных таргетингах.
     *                        См. {@link ru.yandex.direct.core.entity.relevancematch.service.RelevanceMatchAddOperation}.
     * @param autoPriceParams параметры для автоматического вычисления недостающих ставок в бесфразных таргетингах.
     *                        Должен быть не {@code null}, если {@code autoPrices == true}
     */
    private void createRelevanceMatchExecutor(ModelProperty<? super T, List<RelevanceMatch>> relevanceMatchProperty,
                                              RelevanceMatchService relevanceMatchService, Currency clientCurrency,
                                              boolean autoPrices, @Nullable ShowConditionAutoPriceParams autoPriceParams,
                                              long operatorUid, ClientId clientId) {
        SubOperationCreator<RelevanceMatch, AddRelevanceMatchesSubOperation> subOperationCreator =
                relevanceMatches -> new AddRelevanceMatchesSubOperation(relevanceMatchService, clientCurrency,
                        relevanceMatches, autoPrices, autoPriceParams, operatorUid, clientId);
        relevanceMatchExecutor = ListSubOperationExecutor.builder()
                .withFakeParents(complexAdGroups)
                .withChildrenProperty(relevanceMatchProperty)
                .createSubOperationBy(subOperationCreator);
    }

    @Override
    public void prepare(ValidationResult<List<AdGroup>, Defect> adGroupsResult) {
        Set<Long> campaignIds = listToSet(adGroups, AdGroup::getCampaignId);
        List<Campaign> campaigns = campaignRepository.getClientsCampaignsByIds(shard, clientId, campaignIds);
        Map<Long, Campaign> campaignsMap = listToMap(campaigns, Campaign::getId);

        Map<Integer, AdGroupInfoForRelevanceMatchAdd> adGroupInfoByRelevanceMatchIndex = new HashMap<>();
        relevanceMatchExecutor.getIndexMultimap().forEach((adGroupIndex, relevanceMatchIndex) -> {
            Long campaignId = adGroups.get(adGroupIndex).getCampaignId();
            adGroupInfoByRelevanceMatchIndex.put(relevanceMatchIndex,
                    new AdGroupInfoForRelevanceMatchAdd(adGroupIndex, campaignsMap.get(campaignId)));
        });
        relevanceMatchExecutor.getSubOperation().setAdGroupInfoForRelevanceMatchAdd(adGroupInfoByRelevanceMatchIndex);
        relevanceMatchExecutor.prepare(adGroupsResult);
    }

    @Override
    public void apply(MassResult<Long> addAdGroupsResult) {
        Map<Integer, Long> adGroupIdsByRelevanceMatchIndex = new HashMap<>();
        relevanceMatchExecutor.getIndexMultimap().forEach((adGroupIndex, relevanceMatchIndex) ->
                adGroupIdsByRelevanceMatchIndex
                        .put(relevanceMatchIndex, addAdGroupsResult.get(adGroupIndex).getResult())
        );
        relevanceMatchExecutor.getSubOperation().setAdGroupsIds(adGroupIdsByRelevanceMatchIndex);
        relevanceMatchExecutor.apply();
    }
}
