package ru.yandex.direct.core.entity.adgroup.service.complex.suboperation.add;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;

import ru.yandex.direct.core.entity.adgroup.container.ComplexAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.currency.model.cpmyndxfrontpage.CpmYndxFrontpageAdGroupPriceRestrictions;
import ru.yandex.direct.core.entity.retargeting.container.RetargetingAdGroupInfo;
import ru.yandex.direct.core.entity.retargeting.model.TargetInterest;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingService;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionFixedAutoPrices;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.operation.tree.ListSubOperationExecutor;
import ru.yandex.direct.operation.tree.SubOperationCreator;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupModelUtils.checkTargetInterestsConsistency;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class RetargetingLogicSupplier<T extends ComplexAdGroup> extends AddSubEntityLogicSupplier<T> {
    private ListSubOperationExecutor<AdGroup, TargetInterest, AddRetargetingsSubOperation> retargetingsExecutor;
    private final Map<Integer, CpmYndxFrontpageAdGroupPriceRestrictions> cpmYndxFrontpageCurrencyRestrictionsMap;
    private final CampaignRepository campaignRepository;
    private final int shard;

    public RetargetingLogicSupplier(List<T> complexAdGroups,
                                    Map<Integer, CpmYndxFrontpageAdGroupPriceRestrictions> cpmYndxFrontpageCurrencyRestrictionsMap,
                                    ModelProperty<? super T, List<TargetInterest>> targetInterestProperty,
                                    RetargetingService retargetingService, CampaignRepository campaignRepository,
                                    boolean retargetingConditionsNonexistentOnPrepare, boolean autoPrices,
                                    @Nullable ShowConditionFixedAutoPrices fixedAutoPrices,
                                    long operatorUid, ClientId clientId, long clientUid, int shard) {
        super(complexAdGroups);
        this.cpmYndxFrontpageCurrencyRestrictionsMap = cpmYndxFrontpageCurrencyRestrictionsMap;
        this.campaignRepository = campaignRepository;
        this.shard = shard;
        checkTargetInterestsConsistency(complexAdGroups, targetInterestProperty);
        createRetargetingsExecutor(targetInterestProperty, retargetingService,
                retargetingConditionsNonexistentOnPrepare, autoPrices, fixedAutoPrices,
                operatorUid, clientId, clientUid);
    }

    /**
     * @param autoPrices      включает режим автоматического вычисления недостающих ставок в ретаргетингах.
     *                        Cм. {@link ru.yandex.direct.core.entity.retargeting.service.AddRetargetingsOperation}
     * @param fixedAutoPrices Контейнер с фиксированными ставками, которые нужно выставить у ретаргетингов, если ставка
     *                        не была явно указана. Ставки могут быть не для всех ретаргетингов.
     *                        Должен быть не {@code null}, если {@code autoPrices == true}.
     */
    private void createRetargetingsExecutor(ModelProperty<? super T, List<TargetInterest>> targetInterestProperty,
                                            RetargetingService retargetingService,
                                            boolean retargetingConditionsNonexistentOnPrepare,
                                            boolean autoPrices, @Nullable ShowConditionFixedAutoPrices fixedAutoPrices,
                                            long operatorUid, ClientId clientId, long clientUid) {
        SubOperationCreator<TargetInterest, AddRetargetingsSubOperation> subOperationCreator =
                targetInterests -> new AddRetargetingsSubOperation(retargetingService, targetInterests,
                        retargetingConditionsNonexistentOnPrepare,
                        autoPrices, fixedAutoPrices, operatorUid, clientId, clientUid);
        retargetingsExecutor = ListSubOperationExecutor.builder()
                .withFakeParents(complexAdGroups)
                .withChildrenProperty(targetInterestProperty)
                .createSubOperationBy(subOperationCreator);
    }

    public ListSubOperationExecutor<AdGroup, TargetInterest, AddRetargetingsSubOperation> getRetargetingExecutor() {
        return retargetingsExecutor;
    }

    @Override
    public void prepare(ValidationResult<List<AdGroup>, Defect> adGroupsResult) {
        List<Long> campaignsId = mapList(adGroups, AdGroup::getCampaignId);
        Map<Long, CampaignType> campaignsType = campaignRepository.getCampaignsTypeMap(shard, campaignsId);

        Map<Integer, RetargetingAdGroupInfo> retargetingAdGroupInfo = new HashMap<>();
        retargetingsExecutor.getIndexMultimap().forEach((adGroupIndex, retargetingIndex) -> {
            Long campaignId = adGroups.get(adGroupIndex).getCampaignId();
            CampaignType campaignType = campaignsType.get(campaignId);
            AdGroup adGroup = adGroups.get(adGroupIndex);
            retargetingAdGroupInfo.put(retargetingIndex,
                    new RetargetingAdGroupInfo((long) -adGroupIndex, adGroup, campaignId, campaignType,
                            cpmYndxFrontpageCurrencyRestrictionsMap.get(adGroupIndex)));
        });
        retargetingsExecutor.getSubOperation().setRetargetingsAdGroupsInfo(retargetingAdGroupInfo);
        retargetingsExecutor.prepare(adGroupsResult);
    }

    @Override
    public void apply(MassResult<Long> addAdGroupsResult) {
        Map<Integer, Long> adGroupIdsByRetargetingIndex = new HashMap<>();
        retargetingsExecutor.getIndexMultimap().forEach((adGroupIndex, retargetingIndex) -> adGroupIdsByRetargetingIndex
                .put(retargetingIndex, addAdGroupsResult.get(adGroupIndex).getResult()));
        retargetingsExecutor.getSubOperation().setAdGroupsIds(adGroupIdsByRetargetingIndex);
        retargetingsExecutor.apply();
    }
}
