package ru.yandex.direct.core.entity.adgroup.service.complex.suboperation.common;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.sitelink.model.Sitelink;
import ru.yandex.direct.core.entity.sitelink.model.SitelinkSet;
import ru.yandex.direct.core.entity.sitelink.service.SitelinkSetAddOperation;
import ru.yandex.direct.core.entity.sitelink.service.SitelinkSetService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.tree.SubOperation;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.Path;
import ru.yandex.direct.validation.result.PathNode;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static ru.yandex.direct.operation.tree.TreeOperationUtils.prepareNullableOperationAndGetValidationResult;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.concat;
import static ru.yandex.direct.validation.result.PathHelper.index;

public class AddSitelinkSetsSubOperation implements SubOperation<SitelinkSet> {

    private final List<SitelinkSet> externalSitelinkSets;

    private List<SitelinkSet> internalSitelinkSets;
    private List<Map<Integer, Integer>> internalToExternalSitelinksIndexes;
    private SitelinkSetAddOperation sitelinkSetAddOperation;

    private List<Long> sitelinkSetIds;

    public AddSitelinkSetsSubOperation(SitelinkSetService sitelinkSetService,
                                       List<SitelinkSet> sitelinkSets, ClientId clientId) {
        checkArgument(sitelinkSets != null, "sub operation must not be created with null sitelinkSets list");
        this.externalSitelinkSets = sitelinkSets;
        createInternalSitelinkSets();
        createOperation(sitelinkSetService, clientId);
    }

    private void createInternalSitelinkSets() {
        internalSitelinkSets = new ArrayList<>(externalSitelinkSets.size());
        internalToExternalSitelinksIndexes = new ArrayList<>();
        for (SitelinkSet externalSitelinkSet : externalSitelinkSets) {

            Map<Integer, Integer> indexMap = new HashMap<>();

            List<Sitelink> externalSitelinks = externalSitelinkSet.getSitelinks();
            List<Sitelink> internalSitelinks;
            if (externalSitelinks != null) {
                internalSitelinks = new ArrayList<>(externalSitelinks.size());

                int internalSlIndex = 0;
                for (int externalSlIndex = 0; externalSlIndex < externalSitelinks.size(); externalSlIndex++) {
                    Sitelink externalSitelink = externalSitelinks.get(externalSlIndex);
                    if (externalSitelink != null) {
                        internalSitelinks.add(externalSitelink);
                        indexMap.put(internalSlIndex++, externalSlIndex);
                    }
                }
            } else {
                internalSitelinks = null;
            }

            SitelinkSet internalSlSet = new SitelinkSet()
                    .withSitelinks(internalSitelinks);

            internalSitelinkSets.add(internalSlSet);
            internalToExternalSitelinksIndexes.add(indexMap);
        }
    }

    private void createOperation(SitelinkSetService sitelinkSetService, ClientId clientId) {
        sitelinkSetAddOperation = !internalSitelinkSets.isEmpty() ?
                sitelinkSetService.createAddOperation(clientId, internalSitelinkSets, Applicability.FULL) :
                null;
    }

    @Override
    public ValidationResult<List<SitelinkSet>, Defect> prepare() {
        ValidationResult<List<SitelinkSet>, Defect> validationResult =
                prepareNullableOperationAndGetValidationResult(sitelinkSetAddOperation, internalSitelinkSets);
        return validationResult.transformUnchecked(new Transformer());
    }

    @Override
    public void apply() {
        if (sitelinkSetAddOperation != null) {
            MassResult<Long> result = sitelinkSetAddOperation.apply();
            sitelinkSetIds = mapList(result.getResult(), Result::getResult);
            fillIdsInExternalModels();  // todo remove
        } else {
            sitelinkSetIds = emptyList();
        }
    }

    private void fillIdsInExternalModels() {
        //noinspection ResultOfMethodCallIgnored
        StreamEx.zip(externalSitelinkSets, sitelinkSetIds, SitelinkSet::withId).toList();
    }

    public List<SitelinkSet> getSitelinkSets() {
        return internalSitelinkSets;
    }

    @ParametersAreNonnullByDefault
    private class Transformer implements ValidationResult.ValidationResultTransformer<Defect> {

        @Override
        public Map<PathNode, ValidationResult<?, Defect>> transformSubResults(Path path,
                                                                              Map<PathNode, ValidationResult<?, Defect>> subResults) {
            if (!isPathToSitelinks(path)) {
                return ValidationResult.ValidationResultTransformer.super.transformSubResults(path, subResults);
            }

            int sitelinkSetIndex = getSitelinkSetIndex(path);
            checkState(sitelinkSetIndex >= 0 && sitelinkSetIndex < externalSitelinkSets.size(),
                    "validation result for sitelinkSets[%s] is not expected, "
                            + "the size of input sitelinkSets list is %s",
                    sitelinkSetIndex, externalSitelinkSets.size());
            Map<Integer, Integer> indexMap = internalToExternalSitelinksIndexes.get(sitelinkSetIndex);

            Map<PathNode, ValidationResult<?, Defect>> destSubResults = new HashMap<>();
            subResults.forEach(((internalIndexNode, validationResult) -> {

                checkState(internalIndexNode instanceof PathNode.Index,
                        "sitelinks validation result must contain only index path nodes, but field is found: %s",
                        internalIndexNode);

                Integer internalSitelinkIndex = ((PathNode.Index) internalIndexNode).getIndex();
                Integer externalSitelinkIndex = indexMap.get(internalSitelinkIndex);

                checkState(externalSitelinkIndex != null,
                        "external sitelink index is not found for internal sitelinkSets[%s].sitelinks[%s]",
                        sitelinkSetIndex, internalSitelinkIndex);

                destSubResults.put(index(externalSitelinkIndex),
                        transform(concat(path, internalIndexNode), validationResult));
            }));

            return destSubResults;
        }

        private boolean isPathToSitelinks(Path path) {
            if (path.getNodes().size() != 2) {
                return false;
            }
            if (!(path.getNodes().get(0) instanceof PathNode.Index)) {
                return false;
            }
            if (!(path.getNodes().get(1) instanceof PathNode.Field)) {
                return false;
            }

            PathNode.Field sitelinkSetField = (PathNode.Field) path.getNodes().get(1);
            return sitelinkSetField.getName().equals(SitelinkSet.SITELINKS.name());
        }

        private int getSitelinkSetIndex(Path path) {
            return ((PathNode.Index) path.getNodes().get(0)).getIndex();
        }
    }
}
