package ru.yandex.direct.core.entity.adgroup.service.complex.suboperation.common;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Multimap;
import one.util.streamex.EntryStream;
import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.ComplexBidModifier;
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierService;
import ru.yandex.direct.core.entity.bidmodifiers.service.ComplexBidModifierService;
import ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.BidModifierValidationHelper;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.container.CampaignIdAndAdGroupIdPair;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.operation.tree.SubOperation;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.BidModifiersDefects.oneTypeUsedTwiceInComplexModifier;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

/**
 * Массовая операция сохранения корректировок на группах для веб-интерфейса.
 * Сохраняет все переданные наборы, удаляя с групп всё остальное.
 * Удалит все корректировки с группы, если id группы будет в affectedAdGroupIds, но в списке bidModifiers не
 * окажется ни одного набора, принадлежащего этой группе.
 * <p>
 * Использование:
 * <p>
 * Сначала вызов prepare()<br/>
 * Если всё ок (нет ошибок валидации), устанавливаем необходимые данные методом setAdGroupsInfoBeforeApply()<br/>
 * Далее можно вызвать apply()<br/>
 */
@ParametersAreNonnullByDefault
public class SetBidModifiersSubOperation implements SubOperation<ComplexBidModifier> {
    private final BidModifierService bidModifierService;
    private final ComplexBidModifierService complexBidModifierService;

    // Входные данные
    private final ClientId clientId;
    private final long operatorUid;
    private final int shard;
    private final List<ComplexBidModifier> complexBidModifiers;

    private Map<Integer, CampaignType> complexIndexToCampaignTypeMap;

    // Производные, вспомогательные данные (заполняются при вызове prepare())
    private List<BidModifier> bidModifiers;
    private Multimap<Integer, Integer> complexToFlatIndexMap;
    private ValidationResult<List<ComplexBidModifier>, Defect> validationResult;

    // Данные, которые не обязательны для валидации, но должны быть установлены перед сохранением
    private Map<Integer, Long> complexIndexToAdGroupIdMap;  // Для понимания, какому элементу какая группа соответствует
    private Map<Long, Long> adGroupIdToCampaignIdMap;  // Чтобы не ходить в базу за этим соответствием лишний раз
    private Set<Long> affectedAdGroupIds;  // Чтобы узнать, какие группы пришли без корректировок, и очистить их
    private Map<Integer, AdGroup> complexIndexToAdGroupWithTypeMap;

    /**
     * @param complexBidModifiers Список комплексных наборов корректировок. Внутри каждого из корректировок
     *                            можно не указывать campaignId и adGroupId, но обязательно нужно указать type.
     */
    public SetBidModifiersSubOperation(
            ClientId clientId, long operatorUid, int shard,
            BidModifierService bidModifierService,
            ComplexBidModifierService complexBidModifierService,
            List<ComplexBidModifier> complexBidModifiers) {
        // Операция про группы, поэтому гео-корректировок в комплексных наборах быть не должно

        this.complexBidModifierService = complexBidModifierService;
        this.clientId = clientId;
        this.operatorUid = operatorUid;
        this.shard = shard;
        this.bidModifierService = bidModifierService;
        this.complexBidModifiers = complexBidModifiers;
    }

    @Override
    public ValidationResult<List<ComplexBidModifier>, Defect> prepare() {
        checkNotNull(complexIndexToCampaignTypeMap, "complexIndexToCampaignTypeMap should be set before prepare");
        checkNotNull(complexIndexToAdGroupWithTypeMap, "complexIndexToAdGroupWithTypeMap should be set before prepare");

        // Сначала провалидируем корректность структуры complexBidModifiers в целом
        ListValidationBuilder<ComplexBidModifier, Defect> lvb = ListValidationBuilder.of(complexBidModifiers);
        lvb.checkEach(fromPredicate(BidModifierValidationHelper::isExpressionModifiersUsedAtMostOnce,
                oneTypeUsedTwiceInComplexModifier()));
        if (lvb.getResult().hasAnyErrors()) {
            return lvb.getResult();
        }

        // Конвертируем комплексные модели в плоский список с сохранением маппинга индексов
        Pair<List<BidModifier>, Multimap<Integer, Integer>> pair =
                complexBidModifierService.convertFromComplexModelsForAdGroups(complexBidModifiers);
        bidModifiers = pair.getLeft();
        complexToFlatIndexMap = pair.getRight();

        // Обратная мапа для получения типа кампании по индексу в плоском списке
        Map<Integer, Integer> flatToComplexIndexMap = EntryStream.of(complexToFlatIndexMap.asMap())
                .flatMapValues(Collection::stream)
                .collect(toMap(Map.Entry::getValue, Map.Entry::getKey));

        // Валидируем плоский список
        ValidationResult<List<BidModifier>, Defect> flatValidationResult =
                complexBidModifierService.validateBidModifiersFlat(bidModifiers, complexToFlatIndexMap,
                        flatIdx -> complexIndexToCampaignTypeMap.get(flatToComplexIndexMap.get(flatIdx)),
                        flatIdx -> complexIndexToAdGroupWithTypeMap.get(flatToComplexIndexMap.get(flatIdx)),
                        clientId);

        validationResult = lvb.getResult();

        // Переносим результаты валидации на список комплексных моделей
        complexBidModifierService.transferValidationResultFlatToComplex(
                validationResult, flatValidationResult, complexToFlatIndexMap);

        return validationResult;
    }

    /**
     * @param complexIndexToCampaignTypeMap Показывает тип кампании для каждого элемента входного списка,
     */
    public void setCampaignTypesBeforePrepare(Map<Integer, CampaignType> complexIndexToCampaignTypeMap) {
        this.complexIndexToCampaignTypeMap = complexIndexToCampaignTypeMap;
    }

    /**
     * @param complexIndexToAdGroupWithTypeMap Группа с полным типом (включая criterion_type) для
     *                                         каждого элемента входного списка.
     */
    public void setAdGroupWithTypesBeforePrepare(Map<Integer, AdGroup> complexIndexToAdGroupWithTypeMap) {
        this.complexIndexToAdGroupWithTypeMap = complexIndexToAdGroupWithTypeMap;
    }

    /**
     * Установить данные, необходимые для сохранения
     *
     * @param complexIndexToAdGroupIdMap id группы для каждого индекса входного списка
     * @param adGroupIdToCampaignIdMap   соответствие adGroupId → campaignId (необходимо для формирования оптимальных
     *                                   запросов)
     * @param affectedAdGroupIds         список всех групп, затрагиваемых изменениями
     */
    public void setAdGroupsInfoBeforeApply(Map<Integer, Long> complexIndexToAdGroupIdMap,
                                           Map<Long, Long> adGroupIdToCampaignIdMap,
                                           Set<Long> affectedAdGroupIds) {
        checkState(complexIndexToAdGroupIdMap.size() == complexBidModifiers.size());
        checkState(affectedAdGroupIds.containsAll(complexIndexToAdGroupIdMap.values()));
        checkState(adGroupIdToCampaignIdMap.keySet().containsAll(affectedAdGroupIds));
        //
        this.complexIndexToAdGroupIdMap = complexIndexToAdGroupIdMap;
        this.adGroupIdToCampaignIdMap = adGroupIdToCampaignIdMap;
        this.affectedAdGroupIds = affectedAdGroupIds;
    }

    @Override
    public void apply() {
        checkNotNull(validationResult);
        checkState(!validationResult.hasAnyErrors());
        checkNotNull(complexIndexToAdGroupIdMap);
        checkNotNull(adGroupIdToCampaignIdMap);
        checkNotNull(affectedAdGroupIds);

        // Проставляем campaignId и adGroupId везде
        // type проставлять не нужно, его наличие проверяется при конвертации из комплексных моделей в плоские
        for (int idx = 0; idx < complexBidModifiers.size(); idx++) {
            long adGroupId = complexIndexToAdGroupIdMap.get(idx);
            Collection<Integer> flatIndexes = complexToFlatIndexMap.get(idx);
            flatIndexes.forEach(flatIndex -> {
                BidModifier bidModifier = bidModifiers.get(flatIndex);
                bidModifier.setAdGroupId(adGroupId);
                bidModifier.setCampaignId(adGroupIdToCampaignIdMap.get(adGroupId));
            });
        }

        // Убедимся, что campaignId, adGroupId и type установлены на всех корректировках
        checkState(bidModifiers.stream().allMatch(it ->
                it.getCampaignId() != null && it.getAdGroupId() != null && it.getType() != null));

        // Сохраняем изменения
        Set<CampaignIdAndAdGroupIdPair> adGroupsToModify = affectedAdGroupIds.stream()
                .map(adGroupId -> new CampaignIdAndAdGroupIdPair()
                        .withCampaignId(adGroupIdToCampaignIdMap.get(adGroupId))
                        .withAdGroupId(adGroupId))
                .collect(toSet());

        bidModifierService.replaceModifiers(clientId, operatorUid, shard, bidModifiers, adGroupsToModify);
    }
}
