package ru.yandex.direct.core.entity.adgroup.service.complex.suboperation.update;

import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;

import ru.yandex.direct.core.entity.adgroup.container.ComplexAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.repository.KeywordRepository;
import ru.yandex.direct.core.entity.keyword.service.KeywordModifyOperationFactory;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionAutoPriceParams;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.operation.tree.ListSubOperationExecutor;
import ru.yandex.direct.operation.tree.SubOperationCreator;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupModelUtils.checkKeywordsConsistency;

public class KeywordLogicSupplier<T extends ComplexAdGroup> extends UpdateSubEntityLogicSupplier<T> {
    private ListSubOperationExecutor<AdGroup, Keyword, UpdateKeywordListSubOperation> keywordExecutor;
    private ModelProperty<? super T, List<Keyword>> keywordsProperty;

    private final Set<Long> affectedAdGroupIds;

    public KeywordLogicSupplier(List<T> complexAdGroups, ModelProperty<? super T, List<Keyword>> keywordsProperty,
                                Set<Long> affectedAdGroupIds, KeywordModifyOperationFactory keywordModifyOperationFactory,
                                KeywordRepository keywordRepository,
                                boolean showConditionAutoPrices, @Nullable ShowConditionAutoPriceParams showConditionAutoPriceParams,
                                long operatorUid, ClientId clientId, long clientUid, int shard) {
        super(complexAdGroups);
        checkKeywordsConsistency(complexAdGroups, keywordsProperty);
        this.keywordsProperty = keywordsProperty;
        this.affectedAdGroupIds = affectedAdGroupIds;
        createKeywordExecutor(keywordsProperty, keywordModifyOperationFactory, keywordRepository,
                showConditionAutoPrices, showConditionAutoPriceParams, operatorUid, clientId, clientUid, shard);
        fillAdGroupId();

    }

    /**
     * @param showConditionAutoPrices      включает режим автоматического выставления недостающих ставок в условиях
     *                                     показов
     *                                     См. {@link ru.yandex.direct.core.entity.keyword.service.KeywordsModifyOperation}
     * @param showConditionAutoPriceParams параметры для вычисления недостающих ставок в условиях показов.
     *                                     N.B.: это параметр для всех условий показа, не только для КФ, т.к. может
     *                                     произойти копирование группы при переполнении.
     *                                     Должен быть не {@code null}, если {@code showConditionAutoPrices == true}.
     */
    private void createKeywordExecutor(ModelProperty<? super T, List<Keyword>> keywordsProperty,
                                       KeywordModifyOperationFactory keywordModifyOperationFactory, KeywordRepository keywordRepository,
                                       boolean showConditionAutoPrices, @Nullable ShowConditionAutoPriceParams showConditionAutoPriceParams,
                                       long operatorUid, ClientId clientId, long clientUid, int shard) {
        SubOperationCreator<Keyword, UpdateKeywordListSubOperation> subOperationCreator =
                allKeywords -> new UpdateKeywordListSubOperation(keywordModifyOperationFactory, keywordRepository,
                        allKeywords, showConditionAutoPrices, showConditionAutoPriceParams,
                        operatorUid, clientId, clientUid, shard);
        keywordExecutor = ListSubOperationExecutor.builder()
                .withFakeParents(complexAdGroups)
                .withChildrenProperty(keywordsProperty)
                .createSubOperationBy(subOperationCreator);
    }

    private void fillAdGroupId() {
        for (T complexAdGroup : complexAdGroups) {
            Long adGroupId = complexAdGroup.getAdGroup().getId();

            List<Keyword> keywords = keywordsProperty.get(complexAdGroup);
            if (!isEmpty(keywords)) {
                for (Keyword keyword : keywords) {
                    keyword.withAdGroupId(adGroupId);
                }
            }
        }
    }

    @Override
    public void prepare(ValidationResult<List<AdGroup>, Defect> adGroupsResults) {
        keywordExecutor.getSubOperation().setAffectedAdGroupIds(affectedAdGroupIds);
        keywordExecutor.prepare(adGroupsResults);
    }

    @Override
    public void apply(MassResult<Long> updateAdGroupsResult) {
        keywordExecutor.apply();
    }
}
