package ru.yandex.direct.core.entity.adgroup.service.complex.suboperation.update;

import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;

import ru.yandex.direct.core.entity.adgroup.container.ComplexAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatch;
import ru.yandex.direct.core.entity.relevancematch.repository.RelevanceMatchRepository;
import ru.yandex.direct.core.entity.relevancematch.service.RelevanceMatchService;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionAutoPriceParams;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.operation.tree.ListSubOperationExecutor;
import ru.yandex.direct.operation.tree.SubOperationCreator;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupModelUtils.checkRelevanceMatchesConsistency;

public class RelevanceMatchLogicSupplier<T extends ComplexAdGroup> extends UpdateSubEntityLogicSupplier<T> {
    private ListSubOperationExecutor<AdGroup, RelevanceMatch, UpdateRelevanceSubOperation> relevanceMatchExecutor;

    private final Set<Long> affectedAdGroupIds;

    public RelevanceMatchLogicSupplier(List<T> complexAdGroups,
                                       ModelProperty<? super T, List<RelevanceMatch>> relevanceMatchProperty, Set<Long> affectedAdGroupIds,
                                       RelevanceMatchService relevanceMatchService,
                                       RelevanceMatchRepository relevanceMatchRepository,
                                       boolean autoPrices, @Nullable ShowConditionAutoPriceParams autoPriceParams,
                                       long operatorUid, ClientId clientId, long clientUid, int shard) {
        super(complexAdGroups);
        checkRelevanceMatchesConsistency(complexAdGroups, relevanceMatchProperty);
        this.affectedAdGroupIds = affectedAdGroupIds;
        createRelevanceMatchExecutor(relevanceMatchProperty, relevanceMatchService, relevanceMatchRepository,
                autoPrices, autoPriceParams, operatorUid, clientId, clientUid, shard);
        fillAdGroupId(relevanceMatchProperty);
    }

    /**
     * @param autoPrices      включает режим автоматического выставления недостающих ставок в бесфразных таргетингах
     *                        См. {@link ru.yandex.direct.core.entity.relevancematch.service.RelevanceMatchModifyOperation}
     * @param autoPriceParams параметры для вычисления недостающих ставок в бесфразных таргетингах.
     *                        Должен быть не {@code null}, если {@code autoPrices == true}
     */
    private void createRelevanceMatchExecutor(ModelProperty<? super T, List<RelevanceMatch>> relevanceMatchProperty,
                                              RelevanceMatchService relevanceMatchService, RelevanceMatchRepository relevanceMatchRepository,
                                              boolean autoPrices, @Nullable ShowConditionAutoPriceParams autoPriceParams,
                                              long operatorUid, ClientId clientId, long clientUid, int shard) {
        SubOperationCreator<RelevanceMatch, UpdateRelevanceSubOperation> subOperationCreator =
                relevanceMatches -> new UpdateRelevanceSubOperation(relevanceMatchService,
                        relevanceMatchRepository, relevanceMatches, autoPrices, autoPriceParams,
                        operatorUid, clientId, clientUid, shard);
        relevanceMatchExecutor = ListSubOperationExecutor.builder()
                .withFakeParents(complexAdGroups)
                .withChildrenProperty(relevanceMatchProperty)
                .createSubOperationBy(subOperationCreator);
    }

    private void fillAdGroupId(ModelProperty<? super T, List<RelevanceMatch>> relevanceMatchProperty) {
        for (T complexAdGroup : complexAdGroups) {
            Long adGroupId = complexAdGroup.getAdGroup().getId();

            List<RelevanceMatch> relevanceMatches = relevanceMatchProperty.get(complexAdGroup);
            if (!isEmpty(relevanceMatches)) {
                for (RelevanceMatch relevanceMatch : relevanceMatches) {
                    relevanceMatch.withAdGroupId(adGroupId);
                }
            }
        }
    }

    @Override
    public void prepare(ValidationResult<List<AdGroup>, Defect> adGroupsResults) {
        relevanceMatchExecutor.getSubOperation().setAffectedAdGroupIds(affectedAdGroupIds);
        relevanceMatchExecutor.prepare(adGroupsResults);
    }

    @Override
    public void apply(MassResult<Long> updateAdGroupsResult) {
        relevanceMatchExecutor.apply();
    }
}
