package ru.yandex.direct.core.entity.adgroup.service.complex.suboperation.update;

import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;

import ru.yandex.direct.core.entity.adgroup.container.ComplexAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupSimple;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.currency.model.cpmyndxfrontpage.CpmYndxFrontpageAdGroupPriceRestrictions;
import ru.yandex.direct.core.entity.retargeting.model.TargetInterest;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingService;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionFixedAutoPrices;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.operation.tree.ListSubOperationExecutor;
import ru.yandex.direct.operation.tree.SubOperationCreator;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupModelUtils.checkTargetInterestsConsistency;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

public class RetargetingLogicSupplier<T extends ComplexAdGroup> extends UpdateSubEntityLogicSupplier<T> {
    private ListSubOperationExecutor<AdGroup, TargetInterest, UpdateRetargetingsSubOperation> targetInterestExecutor;
    private ModelProperty<? super T, List<TargetInterest>> targetInterestProperty;

    private final Map<Long, AdGroupSimple> adGroupSimpleMap;
    private final Map<Long, CpmYndxFrontpageAdGroupPriceRestrictions> affectedFrontpageAdGroupsPriceRestrictions;

    public RetargetingLogicSupplier(List<T> complexAdGroups,
                                    ModelProperty<? super T, List<TargetInterest>> targetInterestProperty,
                                    RetargetingService retargetingService, Map<Long, AdGroupSimple> adGroupSimpleMap,
                                    Map<Long, CpmYndxFrontpageAdGroupPriceRestrictions> affectedFrontpageAdGroupsPriceRestrictions,
                                    boolean autoPrices, @Nullable ShowConditionFixedAutoPrices fixedAutoPrices, long operatorUid,
                                    ClientId clientId, long clientUid, int shard) {
        super(complexAdGroups);
        checkTargetInterestsConsistency(complexAdGroups, targetInterestProperty);
        this.targetInterestProperty = targetInterestProperty;
        this.adGroupSimpleMap = adGroupSimpleMap;
        this.affectedFrontpageAdGroupsPriceRestrictions = affectedFrontpageAdGroupsPriceRestrictions;
        createTargetInterestExecutor(targetInterestProperty, retargetingService, autoPrices, fixedAutoPrices,
                operatorUid,
                clientId, clientUid, shard);
        fillAdGroupId();
    }

    /**
     * @param autoPrices      включает режим автоматического выставления недостающих ставок в ретаргетингах
     *                        См. {@link ru.yandex.direct.core.entity.retargeting.service.AddRetargetingsOperation}
     * @param fixedAutoPrices Контейнер с фиксированными ставками, которые нужно выставить у ретаргетингов, если ставка
     *                        не была явно указана. Ставки могут быть указаны не для всех ретаргетингов.
     *                        Должен быть не {@code null}, если {@code autoPrices == true}.
     */
    private void createTargetInterestExecutor(ModelProperty<? super T, List<TargetInterest>> targetInterestProperty,
                                              RetargetingService retargetingService,
                                              boolean autoPrices, @Nullable ShowConditionFixedAutoPrices fixedAutoPrices,
                                              long operatorUid, ClientId clientId, long clientUid, int shard) {
        AdGroupType adGroupType = complexAdGroups.get(0).getAdGroup().getType();
        SubOperationCreator<TargetInterest, UpdateRetargetingsSubOperation> subOperationCreator =
                targetInterests -> new UpdateRetargetingsSubOperation(retargetingService,
                        targetInterests, adGroupType, autoPrices, fixedAutoPrices, operatorUid, clientId, clientUid,
                        shard);
        targetInterestExecutor = ListSubOperationExecutor.builder()
                .withFakeParents(complexAdGroups)
                .withChildrenProperty(targetInterestProperty)
                .createSubOperationBy(subOperationCreator);
    }

    private void fillAdGroupId() {
        for (T complexAdGroup : complexAdGroups) {
            Long adGroupId = complexAdGroup.getAdGroup().getId();

            List<TargetInterest> targetInterests = targetInterestProperty.get(complexAdGroup);
            if (!isEmpty(targetInterests)) {
                for (TargetInterest targetInterest : targetInterests) {
                    targetInterest.withAdGroupId(adGroupId);
                }
            }
        }
    }

    @Override
    public void prepare(ValidationResult<List<AdGroup>, Defect> adGroupsResults) {
        targetInterestExecutor.getSubOperation().setAffectedAdGroupsMap(adGroupSimpleMap);
        targetInterestExecutor.getSubOperation()
                .setAffectedAdGroupsPriceRestrictions(affectedFrontpageAdGroupsPriceRestrictions);
        targetInterestExecutor.getSubOperation()
                .setPreparedAdGroupModels(listToMap(adGroupsResults.getValue(), ModelWithId::getId));
        targetInterestExecutor.prepare(adGroupsResults);
    }

    @Override
    public void apply(MassResult<Long> updateAdGroupsResult) {
        targetInterestExecutor.apply();
    }
}
