package ru.yandex.direct.core.entity.adgroup.service.complex.suboperation.update;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;

import one.util.streamex.EntryStream;

import ru.yandex.direct.core.entity.adgroup.model.ContentPromotionAdgroupType;
import ru.yandex.direct.core.entity.banner.model.BannerWithAdGroupId;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.service.BannersAddOperation;
import ru.yandex.direct.core.entity.banner.service.BannersAddOperationFactory;
import ru.yandex.direct.core.entity.banner.service.BannersUpdateOperation;
import ru.yandex.direct.core.entity.banner.service.BannersUpdateOperationFactory;
import ru.yandex.direct.core.entity.banner.service.DatabaseMode;
import ru.yandex.direct.core.entity.banner.service.moderation.ModerationMode;
import ru.yandex.direct.core.entity.banner.service.validation.BannerValidationInfo;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.operation.tree.SubOperation;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.core.entity.adgroup.service.complex.suboperation.update.converter.ComplexBannerUpdateConverter.bannersToCoreModelChanges;
import static ru.yandex.direct.core.entity.banner.service.validation.type.BannerTypeValidationPredicates.isTextBanner;
import static ru.yandex.direct.operation.tree.ItemSubOperationExecutor.extractSubList;
import static ru.yandex.direct.operation.tree.TreeOperationUtils.mergeSubListValidationResults;
import static ru.yandex.direct.operation.tree.TreeOperationUtils.mergeSubListValidationResultsFromAnotherObject;
import static ru.yandex.direct.operation.tree.TreeOperationUtils.preparePartialModelOperationAndGetValidationResult;
import static ru.yandex.direct.utils.FunctionalUtils.mapAndFilterToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Общая сабоперация обновления баннеров без связанных сущностей
 */
public class UpdateBannersSubOperation<T extends BannerWithSystemFields> implements SubOperation<T> {

    private List<T> allBanners;

    private List<BannerWithAdGroupId> bannersToAdd;
    private Map<Integer, Integer> addBannersIndexMap;
    protected Map<Integer, Integer> addTextBannersIndexMap;
    protected BannersAddOperation bannersAddOperation;

    private List<T> bannersToUpdate;
    private Map<Integer, Integer> updateBannersIndexMap;
    protected Map<Integer, Integer> updateTextBannersIndexMap;
    protected BannersUpdateOperation<BannerWithSystemFields> bannersUpdateOperation;

    @Nullable
    private Set<Integer> elementIndexesToApply;

    public UpdateBannersSubOperation(ModerationMode moderationMode, List<T> banners,
                                     BannersAddOperationFactory bannersAddOperationFactory,
                                     BannersUpdateOperationFactory bannersUpdateOperationFactory,
                                     long operatorUid, ClientId clientId, boolean isUcPreValidation,
                                     DatabaseMode databaseMode) {
        checkArgument(banners != null, "sub operation must not be created with empty banners list");

        this.allBanners = banners;

        createBannersAddOperation(moderationMode, bannersAddOperationFactory, mapList(banners, Function.identity()),
                operatorUid, clientId, isUcPreValidation, databaseMode);
        createBannersUpdateOperation(moderationMode, bannersUpdateOperationFactory, banners,
                operatorUid, clientId, isUcPreValidation);
    }

    private void createBannersAddOperation(ModerationMode moderationMode, BannersAddOperationFactory bannersAddOperationFactory,
                                           List<BannerWithAdGroupId> banners,
                                           long operatorUid, ClientId clientId, boolean isUcPreValidation,
                                           DatabaseMode databaseMode) {
        addBannersIndexMap = new HashMap<>();
        bannersToAdd = extractSubList(addBannersIndexMap, banners, banner -> banner.getId() == null);
        addTextBannersIndexMap = EntryStream.of(addBannersIndexMap)
                .filterValues(addedBannerIndex -> isTextBanner(bannersToAdd.get(addedBannerIndex)))
                .toMap();
        if (!bannersToAdd.isEmpty()) {
            this.bannersAddOperation = bannersAddOperationFactory.createPartialAddOperationAsPartOfComplexOperation(
                    bannersToAdd, operatorUid, clientId, moderationMode, isUcPreValidation, databaseMode);
        }
    }

    private void createBannersUpdateOperation(ModerationMode moderationMode,
                                              BannersUpdateOperationFactory bannersUpdateOperationFactory,
                                              List<T> banners, long operatorUid,
                                              ClientId clientId, boolean isUcPreValidation) {
        updateBannersIndexMap = new HashMap<>();
        bannersToUpdate = extractSubList(updateBannersIndexMap, banners, banner -> banner.getId() != null);
        updateTextBannersIndexMap = EntryStream.of(updateBannersIndexMap)
                .filterValues(updatedBannerIndex -> isTextBanner(bannersToUpdate.get(updatedBannerIndex)))
                .toMap();
        if (!bannersToUpdate.isEmpty()) {
            this.bannersUpdateOperation = bannersUpdateOperationFactory
                    .createPartialUpdateOperationAsPartOfComplexOperation(
                            moderationMode, bannersToCoreModelChanges(bannersToUpdate),
                            operatorUid, clientId, isUcPreValidation);
        }
    }

    public void setBannerValidationInfo(Map<Integer, BannerValidationInfo> bannersValidationInfoMap) {
        if (bannersAddOperation != null) {
            bannersAddOperation.setBannerValidationInfo(EntryStream.of(addBannersIndexMap)
                    .mapKeys(bannersValidationInfoMap::get)
                    .invert()
                    .toMap());
        }

        if (bannersUpdateOperation != null) {
            bannersUpdateOperation.setBannerValidationInfo(EntryStream.of(updateBannersIndexMap)
                    .mapKeys(bannersValidationInfoMap::get)
                    .invert()
                    .toMap());
        }
    }

    public void setContentPromotionBannerValidationInfo(
            Map<Integer, ContentPromotionAdgroupType> bannersValidationInfoMap) {
        // При обновлении баннера ContentPromotionAdgroupType группы уже известен
        // Так как он не меняется, прокидывание в операцию обновления не требуется
        if (bannersAddOperation != null) {
            bannersAddOperation.setContentPromotionAdGroupTypeMap(EntryStream.of(addBannersIndexMap)
                    .mapKeys(bannersValidationInfoMap::get)
                    .invert()
                    .toMap());
        }
    }

    public void setElementIndexesToApply(@Nullable Set<Integer> elementIndexesToApply) {
        this.elementIndexesToApply = elementIndexesToApply;
    }

    public ValidationResult<List<T>, Defect> prepare() {
        ValidationResult<List<T>, Defect> validationResult = new ValidationResult<>(allBanners);

        ValidationResult<List<BannerWithAdGroupId>, Defect> addValidationResult =
                preparePartialModelOperationAndGetValidationResult(bannersAddOperation, bannersToAdd);
        ValidationResult<List<T>, Defect> updateValidationResult =
                preparePartialModelOperationAndGetValidationResult(bannersUpdateOperation, bannersToUpdate);

        mergeSubListValidationResultsFromAnotherObject(validationResult, addValidationResult, addBannersIndexMap);
        mergeSubListValidationResults(validationResult, updateValidationResult, updateBannersIndexMap);

        afterBannersPrepare(validationResult);

        return validationResult;
    }

    protected void afterBannersPrepare(ValidationResult<List<T>, Defect> validationResult) {
    }

    public void apply() {
        beforeBannersApply();

        if (bannersAddOperation != null) {
            if (elementIndexesToApply != null) {
                bannersAddOperation.apply(getMappedElementIndexes(elementIndexesToApply, addBannersIndexMap));
            } else {
                bannersAddOperation.apply();
            }
        }

        if (bannersUpdateOperation != null) {
            if (elementIndexesToApply != null) {
                bannersUpdateOperation.apply(getMappedElementIndexes(elementIndexesToApply, updateBannersIndexMap));
            } else {
                bannersUpdateOperation.apply();
            }
        }
    }

    private static Set<Integer> getMappedElementIndexes(Set<Integer> elementIndexes, Map<Integer, Integer> indexMap) {
        return mapAndFilterToSet(elementIndexes, indexMap::get, Objects::nonNull);
    }

    protected void beforeBannersApply() {
    }
}
