package ru.yandex.direct.core.entity.adgroup.service.complex.suboperation.update;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;

import com.google.common.collect.SetMultimap;
import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;

import ru.yandex.direct.core.entity.adgroup.service.complex.suboperation.KeywordSeparationUtils;
import ru.yandex.direct.core.entity.adgroup.service.complex.suboperation.update.converter.KeywordUpdateConverter;
import ru.yandex.direct.core.entity.keyword.container.KeywordsModificationContainer;
import ru.yandex.direct.core.entity.keyword.container.KeywordsModificationResult;
import ru.yandex.direct.core.entity.keyword.container.KeywordsModifyOperationParams;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.repository.KeywordRepository;
import ru.yandex.direct.core.entity.keyword.service.KeywordModifyOperationFactory;
import ru.yandex.direct.core.entity.keyword.service.KeywordsModifyOperation;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionAutoPriceParams;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.tree.SubOperation;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.core.entity.keyword.container.KeywordsModificationContainer.ADD_LIST_NAME;
import static ru.yandex.direct.core.entity.keyword.container.KeywordsModificationContainer.UPDATE_LIST_NAME;
import static ru.yandex.direct.core.entity.keyword.container.KeywordsModificationContainer.addUpdateDelete;
import static ru.yandex.direct.core.entity.keyword.model.Keyword.PHRASE;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.keyphrase.PhraseDefects.incorrectUseOfParenthesis;
import static ru.yandex.direct.operation.tree.ItemSubOperationExecutor.extractSubList;
import static ru.yandex.direct.operation.tree.TreeOperationUtils.mergeSubListValidationResults;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.index;

public class UpdateKeywordListSubOperation implements SubOperation<Keyword> {

    private final KeywordModifyOperationFactory keywordModifyOperationFactory;
    private final KeywordRepository keywordRepository;

    private final List<Keyword> sourceKeywords;
    private boolean autoPrices;
    private ShowConditionAutoPriceParams showConditionAutoPriceParams;
    private final long operatorUid;
    private final ClientId clientId;
    private final long clientUid;
    private final int shard;

    private Set<Long> affectedAdGroupIds;

    /**
     * Список разделенных по круглым скобкам входных фраз.
     * Обновляемая фраза с круглыми скобками при разделении
     * превращается в одно обновляемую и остальные добавляемые.
     */
    private List<Keyword> separatedKeywords = new ArrayList<>();

    /**
     * Список индексов ключевых фраз с ошибками использования круглых скобок в исходном списке
     */
    private List<Integer> sourceKeywordIndexesWithParenthesisError = new ArrayList<>();

    /**
     * Индекс элемента соответствует индексу в списке разделенных
     * по круглым скобкам фраз - {@link #separatedKeywords}.
     * <p>
     * Значение элемента означает индекс в исходном списке фраз = {@link #sourceKeywords}.
     * <p>
     * Таким образом, получение индекса во входном списке
     * ключевых фраз по индексу в разделенном списке:
     * {@code sourceKeywordIndexes.get(separatedIndex)}.
     */
    private List<Integer> sourceKeywordIndexes = new ArrayList<>();

    /**
     * Список ключевиков на добавление (с предварительным разделением по круглым скобкам)
     */
    private List<Keyword> keywordsToAdd;

    /**
     * Ключ - индекс из списка разделенных ключевых фраз - {@link #separatedKeywords}.
     * Значение - индекс из списка добавляемых ключевых фраз - {@link #keywordsToAdd}.
     */
    private Map<Integer, Integer> addIndexMap;

    /**
     * Список изменений ключевиков для обновления (с предварительным разделением по круглым скобкам)
     */
    private List<ModelChanges<Keyword>> keywordChangesToUpdate;

    /**
     * Ключ - индекс из списка разделенных ключевых фраз - {@link #separatedKeywords}.
     * Значение - индекс из списка изменений ключевых фраз - {@link #keywordChangesToUpdate}.
     */
    private Map<Integer, Integer> updateIndexMap;

    /**
     * Список id на удаление.
     */
    private List<Long> keywordIdsToDelete;

    private KeywordsModifyOperation keywordsModifyOperation;

    /**
     * @param showConditionAutoPrices      включает режим автоматического выставления недостающих ставок в условиях
     *                                     показов.
     *                                     См. {@link ru.yandex.direct.core.entity.keyword.service.KeywordsModifyOperation}
     * @param showConditionAutoPriceParams параметры для вычисления недостающих ставок в условиях показов.
     *                                     N.B.: это параметр для всех условий показа, не только для КФ, т.к. может
     *                                     произойти копирование группы при переполнении.
     *                                     Должен быть не {@code null}, если {@code showConditionAutoPrices == true}
     */
    public UpdateKeywordListSubOperation(
            KeywordModifyOperationFactory keywordModifyOperationFactory,
            KeywordRepository keywordRepository,
            List<Keyword> sourceKeywords,
            boolean showConditionAutoPrices, @Nullable ShowConditionAutoPriceParams showConditionAutoPriceParams,
            long operatorUid,
            ClientId clientId,
            long clientUid,
            int shard) {
        checkArgument(sourceKeywords != null,
                "sub operation must not be created with null relevanceMatch list");
        this.keywordModifyOperationFactory = keywordModifyOperationFactory;
        this.keywordRepository = keywordRepository;
        this.sourceKeywords = sourceKeywords;
        this.autoPrices = showConditionAutoPrices;
        this.showConditionAutoPriceParams = showConditionAutoPriceParams;
        this.operatorUid = operatorUid;
        this.clientId = clientId;
        this.clientUid = clientUid;
        this.shard = shard;
    }

    public void setAffectedAdGroupIds(Set<Long> affectedAdGroupIds) {
        this.affectedAdGroupIds = affectedAdGroupIds;
    }

    public ValidationResult<List<Keyword>, Defect> prepare() {
        // разбиваем входной список ключевиков по круглым скобкам
        KeywordSeparationUtils.separateKeywords(sourceKeywords, separatedKeywords,
                sourceKeywordIndexes, sourceKeywordIndexesWithParenthesisError);
        if (hasKeywordSeparationErrors()) {
            // если присутствуют ошибки использования круглых скобок,
            // то собираем результат валидации вручную, не вызывая операцию
            ValidationResult<List<Keyword>, Defect> vr = new ValidationResult<>(sourceKeywords);
            for (int sourceIndexWithError : sourceKeywordIndexesWithParenthesisError) {
                Keyword keyword = sourceKeywords.get(sourceIndexWithError);
                ValidationResult<Keyword, Defect> keywordVr =
                        vr.getOrCreateSubValidationResult(index(sourceIndexWithError), keyword);
                ValidationResult<String, Defect> phraseVr =
                        keywordVr.getOrCreateSubValidationResult(field(PHRASE), keyword.getPhrase());
                phraseVr.addError(incorrectUseOfParenthesis());
            }
            return vr;
        }

        fillAddAndUpdateLists();
        fillDeleteList();
        createModifyKeywordsOperation();

        ValidationResult<List<Keyword>, Defect> sourceValidationResult = new ValidationResult<>(sourceKeywords);

        if (keywordsModifyOperation == null) {
            return sourceValidationResult;
        }

        Optional<Result<KeywordsModificationResult>> resultOptional = keywordsModifyOperation.prepare();
        if (!resultOptional.isPresent()) {
            return sourceValidationResult;
        }

        mergeValidationResults(resultOptional.get(), sourceValidationResult);
        return sourceValidationResult;
    }

    /**
     * @return true, если присутствуют ошибки разделения фраз по круглым скобкам и операция ядра не будет вызвана
     */
    private boolean hasKeywordSeparationErrors() {
        return !sourceKeywordIndexesWithParenthesisError.isEmpty();
    }

    private void fillAddAndUpdateLists() {
        addIndexMap = new HashMap<>();
        updateIndexMap = new HashMap<>();
        keywordsToAdd = extractSubList(addIndexMap, separatedKeywords, k -> k.getId() == null);
        keywordChangesToUpdate = extractSubList(updateIndexMap, separatedKeywords,
                k -> k.getId() != null, KeywordUpdateConverter::keywordToModelChanges);
    }

    private void fillDeleteList() {
        checkState(affectedAdGroupIds != null, "affected adGroup ids must be set before prepare()");

        SetMultimap<Long, Long> adGroupIdsToKeywordsIds =
                keywordRepository.getKeywordIdsByAdGroupIds(shard, clientId, affectedAdGroupIds);
        Set<Long> allKeywordIdsSet = EntryStream.of(adGroupIdsToKeywordsIds.asMap())
                .values()
                .flatCollection(Function.identity())
                .toSet();
        Set<Long> updatedKeywordIds = listToSet(keywordChangesToUpdate, ModelChanges::getId);
        Set<Long> keywordIdsToDelete = Sets.difference(allKeywordIdsSet, updatedKeywordIds);
        this.keywordIdsToDelete = new ArrayList<>(keywordIdsToDelete);
    }

    private void createModifyKeywordsOperation() {
        if (!keywordsToAdd.isEmpty() || !keywordChangesToUpdate.isEmpty() || !keywordIdsToDelete.isEmpty()) {
            KeywordsModificationContainer container =
                    addUpdateDelete(keywordsToAdd, keywordChangesToUpdate, keywordIdsToDelete);
            KeywordsModifyOperationParams operationParams = KeywordsModifyOperationParams.builder()
                    .withAutoPrices(autoPrices)
                    .build();
            keywordsModifyOperation = keywordModifyOperationFactory.createKeywordsModifyOperation(operationParams,
                    container, showConditionAutoPriceParams, operatorUid, clientId, clientUid);
        }
    }

    private void mergeValidationResults(Result<KeywordsModificationResult> result,
                                        ValidationResult<List<Keyword>, Defect> sourceValidationResult) {
        ValidationResult<?, Defect> entireValidationResult = result.getValidationResult();
        //noinspection unchecked
        ValidationResult<List<Keyword>, Defect> addValidationResult =
                (ValidationResult<List<Keyword>, Defect>)
                        entireValidationResult.getSubResults().get(field(ADD_LIST_NAME));
        //noinspection unchecked
        ValidationResult<List<Keyword>, Defect> updateValidationResult =
                (ValidationResult<List<Keyword>, Defect>)
                        entireValidationResult.getSubResults().get(field(UPDATE_LIST_NAME));

        ValidationResult<List<Keyword>, Defect> separatedValidationResult =
                new ValidationResult<>(separatedKeywords);
        mergeSubListValidationResults(separatedValidationResult, addValidationResult, addIndexMap);
        mergeSubListValidationResults(separatedValidationResult, updateValidationResult, updateIndexMap);

        KeywordSeparationUtils.separatedValidationResultToSourceValidationResult(
                separatedValidationResult, sourceValidationResult, sourceKeywordIndexes);
    }

    public void apply() {
        checkState(!hasKeywordSeparationErrors(),
                "UpdateKeywordListSubOperation.apply() must not be called when error in keywords parenthesis is found");
        if (keywordsModifyOperation != null) {
            keywordsModifyOperation.apply();
        }
    }
}
