package ru.yandex.direct.core.entity.adgroup.service.complex.suboperation.update;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;

import com.google.common.collect.Multimap;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.adgroup.service.complex.suboperation.update.converter.RelevanceMatchUpdateConverter;
import ru.yandex.direct.core.entity.relevancematch.container.RelevanceMatchModification;
import ru.yandex.direct.core.entity.relevancematch.container.RelevanceMatchModificationResult;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatch;
import ru.yandex.direct.core.entity.relevancematch.repository.RelevanceMatchRepository;
import ru.yandex.direct.core.entity.relevancematch.service.RelevanceMatchModifyOperation;
import ru.yandex.direct.core.entity.relevancematch.service.RelevanceMatchService;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionAutoPriceParams;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.tree.SubOperation;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.core.entity.relevancematch.container.RelevanceMatchModification.RELEVANCE_MATCH_ADD;
import static ru.yandex.direct.core.entity.relevancematch.container.RelevanceMatchModification.RELEVANCE_MATCH_UPDATE;
import static ru.yandex.direct.operation.tree.ItemSubOperationExecutor.extractSubList;
import static ru.yandex.direct.operation.tree.TreeOperationUtils.mergeSubListValidationResults;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.validation.result.PathHelper.field;

public class UpdateRelevanceSubOperation implements SubOperation<RelevanceMatch> {

    private final RelevanceMatchService relevanceMatchService;
    private final RelevanceMatchRepository relevanceMatchRepository;

    private final List<RelevanceMatch> relevanceMatches;
    private final boolean autoPrices;
    private final ShowConditionAutoPriceParams showConditionAutoPriceParams;
    private final long operatorUid;
    private final ClientId clientId;
    private final long clientUid;
    private final int shard;

    private Set<Long> affectedAdGroupIds;

    private Map<Integer, Integer> addIndexMap = new HashMap<>();
    private List<RelevanceMatch> relevanceMatchesToAdd = new ArrayList<>();
    private Map<Integer, Integer> updateIndexMap = new HashMap<>();
    private List<ModelChanges<RelevanceMatch>> relevanceMatchesToUpdate = new ArrayList<>();
    private List<Long> relevanceMatchIdsToDelete;

    private RelevanceMatchModifyOperation relevanceMatchModifyOperation;

    /**
     * @param autoPrices      включает режим автоматического выставления недостающих ставок в бесфразных таргетингах
     *                        См. {@link RelevanceMatchModifyOperation}
     * @param autoPriceParams параметры для вычисления недостающих ставок в бесфразных таргетингах.
     *                        Должен быть не {@code null}, если {@code autoPrices == true}
     */
    public UpdateRelevanceSubOperation(
            RelevanceMatchService relevanceMatchService,
            RelevanceMatchRepository relevanceMatchRepository,
            List<RelevanceMatch> relevanceMatches,
            boolean autoPrices, @Nullable ShowConditionAutoPriceParams autoPriceParams,
            long operatorUid, ClientId clientId, long clientUid, int shard) {
        this.relevanceMatchService = relevanceMatchService;
        this.relevanceMatchRepository = relevanceMatchRepository;
        this.relevanceMatches = relevanceMatches;
        this.autoPrices = autoPrices;
        this.showConditionAutoPriceParams = autoPriceParams;
        this.operatorUid = operatorUid;
        this.clientId = clientId;
        this.clientUid = clientUid;
        this.shard = shard;
    }

    public void setAffectedAdGroupIds(Set<Long> affectedAdGroupIds) {
        this.affectedAdGroupIds = affectedAdGroupIds;
    }

    @Override
    public ValidationResult<List<RelevanceMatch>, Defect> prepare() {
        fillAddAndUpdateLists();
        fillDeleteList();
        createOperation();

        ValidationResult<List<RelevanceMatch>, Defect> validationResult =
                new ValidationResult<>(relevanceMatches);

        if (relevanceMatchModifyOperation == null) {
            return validationResult;
        }

        Optional<Result<RelevanceMatchModificationResult>> resultOptional = relevanceMatchModifyOperation.prepare();
        if (!resultOptional.isPresent()) {
            return validationResult;
        }

        mergeValidationResults(resultOptional.get(), validationResult);
        return validationResult;
    }

    private void fillAddAndUpdateLists() {
        addIndexMap = new HashMap<>();
        updateIndexMap = new HashMap<>();
        relevanceMatchesToAdd = extractSubList(addIndexMap, relevanceMatches, r -> r.getId() == null);
        relevanceMatchesToUpdate = extractSubList(updateIndexMap, relevanceMatches,
                r -> r.getId() != null, RelevanceMatchUpdateConverter::relevanceMatchToModelChanges);
    }

    private void fillDeleteList() {
        checkState(affectedAdGroupIds != null, "affected adGroup ids must be set before prepare()");

        Multimap<Long, Long> relevanceMatchesByAdGroupIds =
                relevanceMatchRepository.getRelevanceMatchIdsByAdGroupIds(shard, clientId, affectedAdGroupIds);
        Set<Long> relevanceMathIdsToUpdate = listToSet(relevanceMatchesToUpdate, ModelChanges::getId);

        relevanceMatchIdsToDelete = EntryStream.of(relevanceMatchesByAdGroupIds.asMap())
                .flatMapValues(StreamEx::of)
                .removeValues(relevanceMathIdsToUpdate::contains)
                .values()
                .toList();
    }

    private void createOperation() {
        if (!relevanceMatchesToAdd.isEmpty() || !relevanceMatchesToUpdate.isEmpty() ||
                !relevanceMatchIdsToDelete.isEmpty()) {
            RelevanceMatchModification relevanceMatchModification = new RelevanceMatchModification()
                    .withRelevanceMatchAdd(relevanceMatchesToAdd)
                    .withRelevanceMatchUpdate(relevanceMatchesToUpdate)
                    .withRelevanceMatchIdsDelete(relevanceMatchIdsToDelete);

            relevanceMatchModifyOperation =
                    relevanceMatchService.createFullModifyOperation(clientId, clientUid, operatorUid,
                            relevanceMatchModification, autoPrices, showConditionAutoPriceParams);
        }
    }

    private void mergeValidationResults(Result<RelevanceMatchModificationResult> result,
                                        ValidationResult<List<RelevanceMatch>, Defect> destValidationResult) {
        ValidationResult<?, Defect> entireValidationResult = result.getValidationResult();
        //noinspection unchecked
        ValidationResult<List<RelevanceMatch>, Defect> validationResultsAdd =
                (ValidationResult<List<RelevanceMatch>, Defect>)
                        entireValidationResult.getSubResults().get(field(RELEVANCE_MATCH_ADD.name()));
        //noinspection unchecked
        ValidationResult<List<RelevanceMatch>, Defect> validationResultsUpdate =
                (ValidationResult<List<RelevanceMatch>, Defect>)
                        entireValidationResult.getSubResults().get(field(RELEVANCE_MATCH_UPDATE.name()));

        mergeSubListValidationResults(destValidationResult, validationResultsAdd, addIndexMap);
        mergeSubListValidationResults(destValidationResult, validationResultsUpdate, updateIndexMap);
    }

    @Override
    public void apply() {
        if (relevanceMatchModifyOperation != null) {
            relevanceMatchModifyOperation.apply();
        }
    }
}
