package ru.yandex.direct.core.entity.adgroup.service.complex.suboperation.update;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupSimple;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.service.complex.suboperation.update.converter.RetargetingsUpdateConverter;
import ru.yandex.direct.core.entity.currency.model.cpmyndxfrontpage.CpmYndxFrontpageAdGroupPriceRestrictions;
import ru.yandex.direct.core.entity.retargeting.model.InterestLink;
import ru.yandex.direct.core.entity.retargeting.model.Retargeting;
import ru.yandex.direct.core.entity.retargeting.model.TargetInterest;
import ru.yandex.direct.core.entity.retargeting.service.AddRetargetingsOperation;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingDeleteOperation;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingService;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingUpdateOperation;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingUtils;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionFixedAutoPrices;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.tree.SubOperation;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.Path;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static ru.yandex.direct.operation.tree.ItemSubOperationExecutor.extractSubList;
import static ru.yandex.direct.operation.tree.TreeOperationUtils.mergeSubListValidationResults;
import static ru.yandex.direct.operation.tree.TreeOperationUtils.prepareNullableOperationAndGetValidationResult;

/**
 * Добавляет несуществующие (id = null) ретаргетинги. Удаляет все те, что были привязаны к группе ранее,
 * но в саб операцию не пришли. Обновляет все остальные.
 * Для текстовых групп обновление отключено, только добавление и удаление.
 */
public class UpdateRetargetingsSubOperation implements SubOperation<TargetInterest> {

    private final RetargetingService retargetingService;

    private final List<TargetInterest> targetInterests;
    private final AdGroupType adGroupType;
    private final boolean autoPrices;
    @Nullable
    private final ShowConditionFixedAutoPrices showConditionFixedAutoPrices;
    private final long operatorUid;
    private final ClientId clientId;
    private final long clientUid;
    private final int shard;

    private Map<Long, AdGroupSimple> affectedAdGroupsMap;
    private Map<Long, AdGroup> preparedAdGroupModels;
    private Map<Long, CpmYndxFrontpageAdGroupPriceRestrictions> affectedAdGroupsPriceRestrictions;

    private AddRetargetingsOperation addOperation;
    private Map<Integer, Integer> addIndexMap;

    private RetargetingUpdateOperation updateOperation;
    private Map<Integer, Integer> updateIndexMap;
    private List<TargetInterest> targetInterestsToUpdate;
    private List<Retargeting> retargetingsToUpdate;

    private RetargetingDeleteOperation deleteOperation;

    private List<TargetInterest> virtualExistingTargetInterests;

    /**
     * @param autoPrices      включает режим автоматического выставления недостающих ставок в ретаргетингах
     *                        См. {@link ru.yandex.direct.core.entity.retargeting.service.AddRetargetingsOperation}
     * @param fixedAutoPrices контейнер с фиксированными ставками для автотаргетингов. Должен быть не {@code null},
     *                        если {@code autoPrices == true}
     */
    public UpdateRetargetingsSubOperation(
            RetargetingService retargetingService,
            List<TargetInterest> targetInterests, AdGroupType adGroupType,
            boolean autoPrices, @Nullable ShowConditionFixedAutoPrices fixedAutoPrices,
            long operatorUid, ClientId clientId, long clientUid, int shard) {
        checkNotNull(adGroupType);
        this.retargetingService = retargetingService;
        this.targetInterests = targetInterests;
        this.adGroupType = adGroupType;
        this.autoPrices = autoPrices;
        this.showConditionFixedAutoPrices = fixedAutoPrices;
        this.operatorUid = operatorUid;
        this.clientId = clientId;
        this.clientUid = clientUid;
        this.shard = shard;
    }

    public void setAffectedAdGroupsMap(Map<Long, AdGroupSimple> affectedAdGroupsMap) {
        this.affectedAdGroupsMap = affectedAdGroupsMap;
    }

    public void setPreparedAdGroupModels(Map<Long, AdGroup> preparedAdGroupModels) {
        this.preparedAdGroupModels = preparedAdGroupModels;
    }

    public void setAffectedAdGroupsPriceRestrictions(
            Map<Long, CpmYndxFrontpageAdGroupPriceRestrictions> affectedAdGroupsPriceRestrictions) {
        this.affectedAdGroupsPriceRestrictions = affectedAdGroupsPriceRestrictions;
    }

    @Override
    public ValidationResult<List<TargetInterest>, Defect> prepare() {
        createVirtualExistingTargetInterestList();
        createAddOperation();
        createUpdateOperation();
        createDeleteOperation(virtualExistingTargetInterests);

        prepareDeleteOperation();

        ValidationResult<List<TargetInterest>, Defect> addValidationResult = prepareAddOperation();
        ValidationResult<List<TargetInterest>, Defect> updateValidationResult = prepareUpdateOperation();
        addValidationResult.merge(updateValidationResult);
        return addValidationResult;
    }

    private void createVirtualExistingTargetInterestList() {
        checkState(affectedAdGroupsMap != null, "affectedAdGroupsMap must be set before prepare()");
        virtualExistingTargetInterests = retargetingService
                .getTargetInterestsWithInterestByAdGroupIds(affectedAdGroupsMap.keySet(), clientId, shard);
    }

    private void createAddOperation() {
        addIndexMap = new HashMap<>();
        List<TargetInterest> targetInterestsToAdd =
                extractSubList(addIndexMap, targetInterests, ti -> ti.getId() == null);
        if (!targetInterestsToAdd.isEmpty()) {
            addOperation = retargetingService.createAddOperation(Applicability.FULL, false, false,
                    targetInterestsToAdd, autoPrices, showConditionFixedAutoPrices, operatorUid, clientId, clientUid);
        }
    }

    private void createUpdateOperation() {
        updateIndexMap = new HashMap<>();
        targetInterestsToUpdate =
                extractSubList(updateIndexMap, targetInterests, ti -> ti.getId() != null);
        if (adGroupType == AdGroupType.BASE) {
            return;
        }
        if (!targetInterestsToUpdate.isEmpty()) {
            List<InterestLink> existingInterestLinks = retargetingService.getExistingInterest(shard, clientId);

            retargetingsToUpdate = RetargetingUtils
                    .convertTargetInterestsToRetargetings(targetInterestsToUpdate, existingInterestLinks);
            List<ModelChanges<Retargeting>> modelChanges =
                    RetargetingsUpdateConverter.retargetingsToCoreModelChanges(retargetingsToUpdate);
            updateOperation = retargetingService
                    .createUpdateOperation(Applicability.FULL, modelChanges, autoPrices, showConditionFixedAutoPrices,
                            true, operatorUid, clientId, clientUid);
        }
    }

    private void createDeleteOperation(List<TargetInterest> existingTargetInterests) {
        Set<Long> requestTargetInterestIds = StreamEx.of(targetInterests)
                .map(TargetInterest::getId)
                .nonNull()
                .toSet();
        List<Long> targetInterestIdsToDelete = StreamEx.of(existingTargetInterests)
                .map(TargetInterest::getId)
                .remove(requestTargetInterestIds::contains)
                .toList();
        if (!targetInterestIdsToDelete.isEmpty()) {
            deleteOperation = retargetingService.createDeleteOperation(Applicability.FULL,
                    targetInterestIdsToDelete, operatorUid, clientId, clientUid);
        }
    }

    private void prepareDeleteOperation() {
        if (deleteOperation == null) {
            return;
        }

        Optional<MassResult<Long>> deleteResult = deleteOperation.prepare();
        checkState(!deleteResult.isPresent(), "RetargetingsDeleteOperation must be prepared successfully");

        Set<Long> potentiallyDeletedIds = deleteOperation.getPotentiallyDeletedRetargetingIds();
        virtualExistingTargetInterests.removeIf(ti -> potentiallyDeletedIds.contains(ti.getId()));
    }

    private ValidationResult<List<TargetInterest>, Defect> prepareAddOperation() {
        ValidationResult<List<TargetInterest>, Defect> validationResult =
                new ValidationResult<>(targetInterests);

        if (addOperation == null) {
            return validationResult;
        }

        addOperation.setExistentAdGroupsInfo(affectedAdGroupsMap, virtualExistingTargetInterests,
                affectedAdGroupsPriceRestrictions, preparedAdGroupModels);
        Optional<MassResult<Long>> addResult = addOperation.prepare();

        if (!addResult.isPresent()) {
            return validationResult;
        }

        //noinspection unchecked
        ValidationResult<List<TargetInterest>, Defect> addValidationResult =
                (ValidationResult<List<TargetInterest>, Defect>)
                        addResult.get().getValidationResult();

        mergeSubListValidationResults(validationResult, addValidationResult, addIndexMap);
        return validationResult;
    }

    private ValidationResult<List<TargetInterest>, Defect> prepareUpdateOperation() {
        if (updateOperation != null) {
            updateOperation.setExistentAdGroupsInfoAndFrontpagePriceRestrictions(affectedAdGroupsMap,
                    affectedAdGroupsPriceRestrictions, preparedAdGroupModels);
        }
        ValidationResult<List<Retargeting>, Defect> retargetingsValidationResult =
                prepareNullableOperationAndGetValidationResult(updateOperation, retargetingsToUpdate);

        ValidationResult<List<TargetInterest>, Defect> targetInterestsValidationResult =
                retargetingsValidationResult.transformUnchecked(new RetargetingsValidationResultTransformer());

        ValidationResult<List<TargetInterest>, Defect> allTargetInterestsValidationResult =
                new ValidationResult<>(targetInterests);
        mergeSubListValidationResults(allTargetInterestsValidationResult, targetInterestsValidationResult,
                updateIndexMap);
        return allTargetInterestsValidationResult;
    }

    @Override
    public void apply() {
        if (deleteOperation != null) {
            deleteOperation.apply();
        }
        if (addOperation != null) {
            addOperation.apply();
        }
        if (updateOperation != null) {
            updateOperation.apply();
        }
    }

    @ParametersAreNonnullByDefault
    private class RetargetingsValidationResultTransformer implements ValidationResult.ValidationResultTransformer<Defect> {
        @Override
        public <OV> Object transformValue(Path path, @Nullable OV oldValue) {
            if (path.equals(new Path(emptyList()))) {
                return targetInterests;
            }
            return oldValue;
        }
    }
}
