package ru.yandex.direct.core.entity.adgroup.service.complex.text;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.Multimap;
import com.google.common.collect.SetMultimap;
import com.google.common.collect.Sets;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.container.ComplexAdGroup;
import ru.yandex.direct.core.entity.adgroup.container.ComplexTextAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.service.complex.UpdateComplexAdGroupValidationService;
import ru.yandex.direct.core.entity.banner.container.ComplexBanner;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerWithAdGroupId;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.repository.ClientRepository;
import ru.yandex.direct.core.entity.keyword.repository.KeywordRepository;
import ru.yandex.direct.core.entity.relevancematch.repository.RelevanceMatchRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.queryrec.model.Language;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupConstraints.complexAdGroupHasWeatherBidModifiers;
import static ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupConstraints.weatherBidModifiersAllowed;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class UpdateComplexTextAdGroupValidationService {

    private final UpdateComplexAdGroupValidationService updateValidationService;
    private final CampaignRepository campaignRepository;
    private final BannerTypedRepository typedRepository;
    private final BannerRelationsRepository relationsRepository;
    private final RelevanceMatchRepository relevanceMatchRepository;
    private final KeywordRepository keywordRepository;
    private final ShardHelper shardHelper;
    private final ClientRepository clientRepository;

    @Autowired
    public UpdateComplexTextAdGroupValidationService(
            UpdateComplexAdGroupValidationService updateValidationService,
            CampaignRepository campaignRepository,
            BannerTypedRepository typedRepository,
            BannerRelationsRepository relationsRepository,
            RelevanceMatchRepository relevanceMatchRepository,
            KeywordRepository keywordRepository, ShardHelper shardHelper,
            ClientRepository clientRepository) {
        this.updateValidationService = updateValidationService;
        this.campaignRepository = campaignRepository;
        this.typedRepository = typedRepository;
        this.relationsRepository = relationsRepository;
        this.relevanceMatchRepository = relevanceMatchRepository;
        this.keywordRepository = keywordRepository;
        this.shardHelper = shardHelper;
        this.clientRepository = clientRepository;
    }

    /**
     * Проводит проверки обновления большой группы, специфичные именно
     * для большой группы, которая содержит подмодели баннеров и т.п.
     * <p>
     * Результат валидации содержит на уровне группы результат валидации списка баннеров,
     * как если бы список баннеров действительно был частью группы. Таким образом,
     * {@link ComplexAdGroup} является лишь условным контейнером.
     */
    public ValidationResult<List<AdGroup>, Defect> validateAdGroups(
            ValidationResult<List<AdGroup>, Defect> adGroupsResult,
            List<ComplexTextAdGroup> complexAdGroups, ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        ValidationData validationData = obtainValidationData(shard, clientId, complexAdGroups);
        Long clientRegionId = clientRepository.getCountryRegionIdByClientId(shard, clientId).orElse(null);

        return new ListValidationBuilder<>(adGroupsResult)
                .checkEachBy(adGroup -> updateValidationService
                        .validateAdGroup(adGroup, validationData.adGroupIdToCampaignLanguageMap,
                                validationData.untouchedBannersByAdGroupIds, clientId, clientRegionId))
                .checkEachBy((index, adGroup) -> {
                    ComplexTextAdGroup complexAdGroup = complexAdGroups.get(index);
                    return validateInterconnections(complexAdGroup, validationData, clientId, clientRegionId);
                })
                .getResult();
    }

    private ValidationResult<AdGroup, Defect> validateInterconnections(ComplexTextAdGroup complexAdGroup,
                                                                       ValidationData validationData,
                                                                       ClientId clientId,
                                                                       Long clientRegionId) {
        ModelItemValidationBuilder<AdGroup> vb = ModelItemValidationBuilder.of(complexAdGroup.getAdGroup());
        List<ComplexBanner> complexBanners = complexAdGroup.getComplexBanners();
        var banners = mapList(complexBanners, ComplexBanner::getBanner);

        vb.check(weatherBidModifiersAllowed(),
                When.isTrue(complexAdGroupHasWeatherBidModifiers(complexAdGroup)));

        vb.list(banners, ComplexTextAdGroup.COMPLEX_BANNERS.name())
                .checkBy(list -> updateValidationService
                        .validateComplexBanners(complexAdGroup.getAdGroup(), complexBanners, banners,
                                validationData.adGroupIdToCampaignLanguageMap,
                                validationData.existingBannerIdsByAdGroupIds,
                                clientId, clientRegionId));

        vb.list(complexAdGroup.getKeywords(), ComplexTextAdGroup.KEYWORDS.name())
                .checkBy(list -> updateValidationService
                        .validateKeywords(complexAdGroup.getAdGroup(), complexAdGroup.getKeywords(),
                                validationData.existingKeywordIdsByAdGroupIds));

        vb.list(complexAdGroup.getRelevanceMatches(), ComplexTextAdGroup.RELEVANCE_MATCHES.name())
                .checkBy(list -> updateValidationService
                        .validateRelevanceMatches(complexAdGroup, complexAdGroup.getRelevanceMatches(),
                                validationData.existingRelevanceIdsByAdGroupIds));

        return vb.getResult();
    }

    private ValidationData obtainValidationData(int shard, ClientId clientId,
                                                List<ComplexTextAdGroup> complexAdGroups) {
        List<Long> adGroupIds = mapList(complexAdGroups, complexAdGroup -> complexAdGroup.getAdGroup().getId());

        Map<Long, Language> adGroupIdToCampaignLanguageMap =
                campaignRepository.getCampaignsLangByAdGroupIds(shard, clientId, adGroupIds);

        Multimap<Long, Long> existingBannerIdsByAdGroupIds =
                relationsRepository.getAdGroupIdToBannerIds(shard, adGroupIds);
        Set<Long> existingBannerIds = new HashSet<>(existingBannerIdsByAdGroupIds.values());
        Set<Long> requestBannerIds = StreamEx.of(complexAdGroups)
                .flatCollection(ComplexTextAdGroup::getComplexBanners)
                .map(ComplexBanner::getBanner)
                .map(Banner::getId)
                .toSet();
        Set<Long> untouchedBannerIds = Sets.difference(existingBannerIds, requestBannerIds);
        // достаются баннеры целиком, так как не хочется использовать
        // интерфейс BannerWithLangFields (его хочется переделать)
        List<BannerWithSystemFields> untouchedBanners = typedRepository
                .getStrictlyFullyFilled(shard, untouchedBannerIds, BannerWithSystemFields.class);
        Map<Long, List<BannerWithSystemFields>> untouchedBannersByAdGroupIds = StreamEx.of(untouchedBanners)
                .groupingBy(BannerWithAdGroupId::getAdGroupId, Collectors.toList());

        Multimap<Long, Long> relevanceIdsByAdGroupIds = relevanceMatchRepository
                .getRelevanceMatchIdsByAdGroupIds(shard, clientId, adGroupIds);

        SetMultimap<Long, Long> keywordIdsByAdGroupIds =
                keywordRepository.getKeywordIdsByAdGroupIds(shard, clientId, adGroupIds);

        return new ValidationData(adGroupIdToCampaignLanguageMap, existingBannerIdsByAdGroupIds,
                untouchedBannersByAdGroupIds, relevanceIdsByAdGroupIds, keywordIdsByAdGroupIds);
    }

    private static class ValidationData {
        private final Map<Long, Language> adGroupIdToCampaignLanguageMap;
        private final Multimap<Long, Long> existingBannerIdsByAdGroupIds;
        private final Map<Long, List<BannerWithSystemFields>> untouchedBannersByAdGroupIds;
        private final Multimap<Long, Long> existingRelevanceIdsByAdGroupIds;
        private final SetMultimap<Long, Long> existingKeywordIdsByAdGroupIds;

        ValidationData(Map<Long, Language> adGroupIdToCampaignLanguageMap,
                       Multimap<Long, Long> existingBannerIdsByAdGroupIds,
                       Map<Long, List<BannerWithSystemFields>> untouchedBannersByAdGroupIds,
                       Multimap<Long, Long> existingRelevanceIdsByAdGroupIds,
                       SetMultimap<Long, Long> existingKeywordIdsByAdGroupIds) {
            this.adGroupIdToCampaignLanguageMap = adGroupIdToCampaignLanguageMap;
            this.existingBannerIdsByAdGroupIds = existingBannerIdsByAdGroupIds;
            this.untouchedBannersByAdGroupIds = untouchedBannersByAdGroupIds;
            this.existingRelevanceIdsByAdGroupIds = existingRelevanceIdsByAdGroupIds;
            this.existingKeywordIdsByAdGroupIds = existingKeywordIdsByAdGroupIds;
        }
    }
}
