package ru.yandex.direct.core.entity.adgroup.service.update;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.jooq.Configuration;

import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.StatusModerate;
import ru.yandex.direct.core.entity.adgroup.model.StatusPostModerate;
import ru.yandex.direct.core.entity.adgroup.model.StatusShowsForecast;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.adgroup.service.ModerationMode;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerWithMulticardSet;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerModerationRepository;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusModerate;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.service.CreativeService;
import ru.yandex.direct.core.entity.moderation.ModerationCheckUtils;
import ru.yandex.direct.core.entity.moderation.repository.sending.BannerMulticardSetSendingRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.GeoTreeFactory;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Objects.requireNonNull;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.bannerstorage.client.Utils.isPerformanceLayoutObsolete;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Базовая реализация интерфейса {@link AdGroupUpdateService} включающая в себя бизнес-логику
 * общую для всех типов групп объявлений
 */
@ParametersAreNonnullByDefault
public abstract class AbstractAdGroupUpdateService implements AdGroupUpdateService {
    protected final DslContextProvider dslContextProvider;
    private final AdGroupType supportedType;
    private final BannerCommonRepository bannerCommonRepository;
    private final BannerModerationRepository bannerModerationRepository;
    private final BannerMulticardSetSendingRepository bannerMulticardSetSendingRepository;
    private final CampaignRepository campaignRepository;
    private final AdGroupRepository adGroupRepository;
    private final CreativeService creativeService;
    private final GeoTreeFactory geoTreeFactory;


    protected AbstractAdGroupUpdateService(
            AdGroupType supportedType,
            DslContextProvider dslContextProvider,
            BannerCommonRepository bannerCommonRepository,
            BannerModerationRepository bannerModerationRepository,
            BannerMulticardSetSendingRepository bannerMulticardSetSendingRepository,
            CampaignRepository campaignRepository,
            AdGroupRepository adGroupRepository,
            CreativeService creativeService,
            GeoTreeFactory geoTreeFactory) {
        this.supportedType = supportedType;
        this.dslContextProvider = dslContextProvider;
        this.bannerCommonRepository = bannerCommonRepository;
        this.bannerModerationRepository = bannerModerationRepository;
        this.bannerMulticardSetSendingRepository = bannerMulticardSetSendingRepository;
        this.campaignRepository = campaignRepository;
        this.adGroupRepository = adGroupRepository;
        this.creativeService = creativeService;
        this.geoTreeFactory = geoTreeFactory;
    }

    private boolean isBannerRequireRemoderation(
            BannerWithSystemFields banner, Set<Long> oldGeoIds, Set<Long> newGeoIds) {
        return banner.getStatusModerate() != BannerStatusModerate.NEW
                && ModerationCheckUtils.isRemoderationRequiredForBannerByGeoChange(
                getGeoTree(), banner.getFlags(), oldGeoIds, newGeoIds);
    }

    private boolean isMulticardSetRequireRemoderation(
            BannerWithSystemFields banner, Set<Long> oldGeoIds, Set<Long> newGeoIds) {
        return banner.getStatusModerate() != BannerStatusModerate.NEW
                && banner instanceof BannerWithMulticardSet
                && ModerationCheckUtils.isRemoderationRequiredForMulticardByGeoChange(
                getGeoTree(), oldGeoIds, newGeoIds);
    }

    @Override
    public AdGroupType getSupportedType() {
        return supportedType;
    }

    protected AdGroupPostUpdateOptions doPrepareForUpdate(AdGroupUpdateData adGroupUpdateData,
                                                          ModerationMode moderationMode) {
        AppliedChanges<? extends AdGroup> adGroupChanges = adGroupUpdateData.getAdGroupChanges();
        AdGroup adGroup = adGroupChanges.getModel();

        AdGroupPostUpdateOptions adGroupPostUpdateOptions = new AdGroupPostUpdateOptions(adGroup);

        boolean wasDraftBeforeUpdate = adGroup.getStatusModerate() == StatusModerate.NEW;
        boolean moderationAllowed = adGroupUpdateData.getCampaign().getStatusModerate() != CampaignStatusModerate.NEW;
        boolean moderateUnconditionally = moderationAllowed && wasDraftBeforeUpdate &&
                moderationMode.isForceModerate();
        boolean moderateIfChanged = moderationAllowed && !wasDraftBeforeUpdate &&
                (moderationMode.isDefault() || moderationMode.isForceModerate());
        boolean forceSaveDraft = moderationMode.isForceSaveDraft();

        if (wasDraftBeforeUpdate) {
            adGroupChanges.modify(AdGroup.STATUS_POST_MODERATE, StatusPostModerate.NO);
        }

        if (forceSaveDraft) {

            adGroupChanges.modify(AdGroup.STATUS_MODERATE, StatusModerate.NEW);
            adGroupChanges.modify(AdGroup.STATUS_POST_MODERATE, StatusPostModerate.NO);

            adGroupPostUpdateOptions.setClearBannersModerationFlags(true);

        } else if (moderateUnconditionally) {

            adGroupChanges.modify(AdGroup.STATUS_MODERATE, StatusModerate.READY);
            adGroupPostUpdateOptions.setAdGroupSentToModeration(true);
            adGroupPostUpdateOptions.setClearBannersModerationFlags(true);

            List<Long> needRemoderateBannerIds = mapList(adGroupUpdateData.getBanners(), Banner::getId);
            if (!needRemoderateBannerIds.isEmpty()) {
                adGroupPostUpdateOptions.setRemoderateBannerIds(needRemoderateBannerIds);
            }

            if (adGroup.getStatusPostModerate() != StatusPostModerate.REJECTED) {
                adGroupChanges.modify(AdGroup.STATUS_POST_MODERATE, StatusPostModerate.NO);
            }

        } else if (moderateIfChanged) {

            if (adGroupChanges.changed(AdGroup.GEO)) {

                Set<Long> oldGeoIds = new HashSet<>(requireNonNull(adGroupChanges.getOldValue(AdGroup.GEO)));
                Set<Long> newGeoIds = new HashSet<>(requireNonNull(adGroupChanges.getNewValue(AdGroup.GEO)));

                // Так как проверяем, если geo изменено, то оба значения старое geo и новое geo должны быть не null
                boolean isRemoderationRequiredByGeoChange =
                        ModerationCheckUtils.isRemoderationRequiredByGeoChange(getGeoTree(), oldGeoIds, newGeoIds);

                if (isRemoderationRequiredByGeoChange) {
                    adGroupChanges.modify(AdGroup.STATUS_MODERATE, StatusModerate.READY);
                    adGroupPostUpdateOptions.setAdGroupSentToModeration(true);
                    if (adGroup.getStatusPostModerate() != StatusPostModerate.REJECTED) {
                        adGroupChanges.modify(AdGroup.STATUS_POST_MODERATE, StatusPostModerate.NO);
                        adGroupPostUpdateOptions.setClearBannersModerationFlags(true);
                    }
                }

                List<Long> needRemoderateBannerIds = adGroupUpdateData.getBanners()
                        .stream()
                        .filter(b -> isBannerRequireRemoderation(b, oldGeoIds, newGeoIds))
                        .map(Banner::getId)
                        .collect(toList());
                if (!needRemoderateBannerIds.isEmpty()) {
                    adGroupPostUpdateOptions.setRemoderateBannerIds(needRemoderateBannerIds);
                    adGroupPostUpdateOptions.setClearBannersModerationFlags(true);
                }
            }
        }

        if (adGroupChanges.changed(AdGroup.GEO)) {
            boolean oldAdGroupGeoFlag = getGeoTree().hasGeoFlagByGeo(adGroupChanges.getOldValue(AdGroup.GEO));
            boolean newAdGroupGeoFlag = getGeoTree().hasGeoFlagByGeo(adGroupChanges.getNewValue(AdGroup.GEO));
            if (oldAdGroupGeoFlag != newAdGroupGeoFlag) {
                adGroupPostUpdateOptions.setNewBannersGeoFlag(newAdGroupGeoFlag);
            }

            Set<Long> oldGeoIds = new HashSet<>(requireNonNull(adGroupChanges.getOldValue(AdGroup.GEO)));
            Set<Long> newGeoIds = new HashSet<>(requireNonNull(adGroupChanges.getNewValue(AdGroup.GEO)));
            List<Long> needRemoderateMulticardSetBannerIds = adGroupUpdateData.getBanners()
                    .stream()
                    .filter(b -> isMulticardSetRequireRemoderation(b, oldGeoIds, newGeoIds))
                    .map(Banner::getId)
                    .collect(toList());
            adGroupPostUpdateOptions.setRemoderateMulticardSetBannerIds(needRemoderateMulticardSetBannerIds);
        }

        adGroupPostUpdateOptions.setUpdateCreativesSumGeo(adGroupChanges.changed(AdGroup.GEO) &&
                adGroup.getType() == AdGroupType.PERFORMANCE);

        if (adGroupChanges.changed(AdGroup.MINUS_KEYWORDS_ID)
                || adGroupChanges.changed(AdGroup.LIBRARY_MINUS_KEYWORDS_IDS)
                || adGroupChanges.changed(AdGroup.GEO)
                || adGroupChanges.changed(AdGroup.TRACKING_PARAMS)
                || adGroupChanges.changed(AdGroup.PAGE_GROUP_TAGS)
                || adGroupChanges.changed(AdGroup.TARGET_TAGS)
                || adGroupChanges.changed(AdGroup.PROJECT_PARAM_CONDITIONS)
                || adGroupChanges.changed(AdGroup.CONTENT_CATEGORIES_RETARGETING_CONDITION_RULES)) {
            adGroupChanges.modify(AdGroup.STATUS_BS_SYNCED, StatusBsSynced.NO);
            adGroupChanges.modifyIfNotChanged(AdGroup.LAST_CHANGE, LocalDateTime::now);
        }

        if (adGroupChanges.changed(AdGroup.GEO) || adGroupChanges.changed(AdGroup.MINUS_KEYWORDS)) {
            adGroupChanges.modify(AdGroup.STATUS_SHOWS_FORECAST, StatusShowsForecast.NEW);
            adGroupPostUpdateOptions.setScheduleForecastRecalc(true);
        }

        // outdoor-группы не отправляем на модерацию (DIRECT-88048)
        if (adGroup.getType() == AdGroupType.CPM_OUTDOOR && adGroup.getStatusModerate() != StatusModerate.NEW
                && adGroup.getStatusModerate() != StatusModerate.YES) {
            adGroupChanges.modify(AdGroup.STATUS_MODERATE, StatusModerate.YES);
            adGroupChanges.modify(AdGroup.STATUS_POST_MODERATE, StatusPostModerate.YES);
        }

        return adGroupPostUpdateOptions;
    }

    /**
     * Предообработать сохраняемые группы объявлений и вернуть набор данных определяющих действия, которые
     * необходимо выполнить после обновления
     */
    private List<AdGroupPostUpdateOptions> prepareForUpdate(List<AdGroupUpdateData> adGroupsUpdateData,
                                                            ModerationMode moderationMode) {
        List<AdGroupPostUpdateOptions> adGroupsPostUpdateOptions = new ArrayList<>(adGroupsUpdateData.size());
        for (AdGroupUpdateData adGroupUpdateData : adGroupsUpdateData) {
            adGroupsPostUpdateOptions.add(doPrepareForUpdate(adGroupUpdateData, moderationMode));
        }
        return adGroupsPostUpdateOptions;
    }

    /**
     * Подготовка к обновлению данных, специфичная для данного типа группы.
     * По умолчанию ничего не делает.
     *
     * @param shard    Шард
     * @param clientId ID клиента
     * @param adGroups Список изменений в группах
     */
    protected void beforeUpdateInTransaction(int shard, ClientId clientId, List<AppliedChanges<AdGroup>> adGroups) {
    }

    /**
     * Действия при сохранении группы, которые должны выполнятся в транзакции
     */
    private void doUpdateInTransaction(
            Configuration config, ClientId clientId, Collection<AppliedChanges<AdGroup>> adGroups) {
        adGroupRepository.updateAdGroups(config, clientId, adGroups);
    }

    /**
     * Выполнить необходимые действия после обновления групп вне транзакции
     */
    private void executePostActions(int shard, ClientId clientId,
                                    List<AdGroupPostUpdateOptions> adGroupsPostUpdateData) {
        // Выполняем обновление гео-флагов и сброс флага status_bs_synced за один вызов (Минимизируем round-trip-ы)
        bannerCommonRepository.resetStatusBsSyncedAndUpdateGeoFlagByIds(
                shard,
                adGroupsPostUpdateData.stream()
                        .filter(adGroup -> adGroup.getNewBannersGeoFlag() != null || adGroup
                                .isResetBannersStatusBsSync())
                        .flatMap(adGroup -> adGroup.getAdGroup()
                                .getBanners()
                                .stream()
                                .map(b -> Pair.of(b.getId(), adGroup.getNewBannersGeoFlag())))
                        // toMap выбрасывает исключение в случае null-а в value
                        .collect(HashMap::new, (m, v) -> m.put(v.getKey(), v.getValue()), HashMap::putAll));

        bannerModerationRepository.requestRemoderation(
                shard,
                adGroupsPostUpdateData.stream()
                        // outdoor-баннеры не переотправляются на модерацию (DIRECT-88048)
                        .filter(adGroup -> adGroup.getAdGroup().getType() != AdGroupType.CPM_OUTDOOR)
                        .flatMap(adGroup -> adGroup.getRemoderateBannerIds().stream())
                        .collect(toList()));

        bannerModerationRepository.clearModerationFlags(
                shard,
                adGroupsPostUpdateData.stream()
                        .filter(AdGroupPostUpdateOptions::isClearBannersModerationFlags)
                        .flatMap(adGroup -> adGroup.getAdGroup().getBanners().stream())
                        .map(Banner::getId)
                        .collect(toList()));

        bannerMulticardSetSendingRepository.requestRemoderation(
                shard,
                adGroupsPostUpdateData.stream()
                        .flatMap(adGroup -> adGroup.getRemoderateMulticardSetBannerIds().stream())
                        .collect(toList()));

        campaignRepository.setAutobudgetForecastDate(
                shard, adGroupsPostUpdateData.stream()
                        .filter(AdGroupPostUpdateOptions::isScheduleForecastRecalc)
                        .map(adGroup -> adGroup.getAdGroup().getCampaignId())
                        .collect(toSet()),
                null
        );

        Set<Long> campaignIdsOfAdGroupsSentToModeration = StreamEx.of(adGroupsPostUpdateData)
                .filter(AdGroupPostUpdateOptions::isAdGroupSentToModeration)
                .map(options -> options.getAdGroup().getCampaignId())
                .toSet();
        campaignRepository.sendRejectedCampaignsToModerate(shard, campaignIdsOfAdGroupsSentToModeration);

        Set<Long> adGroupIds = StreamEx.of(adGroupsPostUpdateData)
                .filter(AdGroupPostUpdateOptions::isUpdateCreativesSumGeo)
                .map(AdGroupPostUpdateOptions::getAdGroup)
                .map(AdGroup::getId)
                .toSet();
        updateCreativesGeo(shard, clientId, adGroupIds);
    }

    private void updateCreativesGeo(int shard, ClientId clientId, Set<Long> adGroupIds) {
        Map<Long, List<Creative>> creativesByAdGroupIds =
                creativeService.getCreativesByPerformanceAdGroups(clientId, adGroupIds);
        List<Long> creativeIds = StreamEx.of(creativesByAdGroupIds.values())
                .flatMap(StreamEx::of)
                .remove(creative -> isPerformanceLayoutObsolete(creative.getLayoutId()))
                .map(Creative::getId)
                .distinct()
                .toList();
        creativeService.setGeoForUnmoderatedCreatives(shard, clientId, creativeIds);
        creativeService.sendRejectedCreativesToModeration(shard, creativeIds);
    }

    @Override
    public final void update(int shard, ClientId clientId, ModerationMode moderationMode,
                             List<AdGroupUpdateData> adGroupsUpdateData) {
        if (adGroupsUpdateData.isEmpty()) {
            return;
        }

        List<AppliedChanges<AdGroup>> adGroups =
                mapList(adGroupsUpdateData, AdGroupUpdateData::getAdGroupChanges);

        List<AdGroupPostUpdateOptions> adGroupsPostUpdateOptions = prepareForUpdate(adGroupsUpdateData, moderationMode);

        checkState(adGroupsPostUpdateOptions.size() == adGroupsUpdateData.size());

        beforeUpdateInTransaction(shard, clientId, adGroups);

        dslContextProvider.ppc(shard)
                .transaction(config -> {
                    doUpdateInTransaction(config, clientId, adGroups);
                    doTypeSpecificUpdateInTransaction(config, clientId, adGroups);
                });

        doTypeSpecificUpdateAfterTransaction(clientId, adGroups);

        executePostActions(shard, clientId, adGroupsPostUpdateOptions);
    }

    /**
     * Выполняет внутри основной транзакции Update, специфичный для данного типа группы.
     * По умолчанию не делает ничего.
     *
     * @param config         Конфигурация для транзакции
     * @param clientId       ID клиента
     * @param appliedChanges Список изменений в группах
     */
    protected void doTypeSpecificUpdateInTransaction(
            @Nonnull Configuration config,
            @Nonnull ClientId clientId,
            @Nonnull List<AppliedChanges<AdGroup>> appliedChanges) {
        // Default - no specific actions
    }

    /**
     * Выполняет после основной транзакции Update, специфичный для данного типа группы.
     * По умолчанию не делает ничего.
     *
     * @param clientId       ID клиента
     * @param appliedChanges Список изменений в группах
     */
    protected void doTypeSpecificUpdateAfterTransaction(
            @Nonnull ClientId clientId,
            @Nonnull List<AppliedChanges<AdGroup>> appliedChanges) {
        // Default - no specific actions
    }

    private GeoTree getGeoTree() {
        return geoTreeFactory.getGlobalGeoTree();
    }
}
