package ru.yandex.direct.core.entity.adgroup.service.update;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.repository.BannerModerationRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.banner.service.text.BannerTextExtractor;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.common.util.GuavaCollectors.toImmutableEnumMap;
import static ru.yandex.direct.core.entity.banner.repository.filter.BannerFilterFactory.bannerAdGroupIdFilter;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@Service
public class AdGroupUpdateServices {
    private final Map<AdGroupType, AdGroupUpdateService> updateServices;
    private final BannerTextExtractor bannerTextExtractor;
    private final BannerTypedRepository bannerRepository;
    private final BannerModerationRepository bannerModerationRepository;

    @Autowired
    public AdGroupUpdateServices(List<AdGroupUpdateService> updateServices,
                                 BannerTextExtractor bannerTextExtractor,
                                 BannerTypedRepository bannerRepository,
                                 BannerModerationRepository bannerModerationRepository) {

        this.updateServices = updateServices.stream()
                .collect(
                        toImmutableEnumMap(
                                AdGroupType.class,
                                AdGroupUpdateService::getSupportedType,
                                Function.identity()));
        this.bannerTextExtractor = bannerTextExtractor;
        this.bannerRepository = bannerRepository;
        this.bannerModerationRepository = bannerModerationRepository;
    }

    public AdGroupUpdateService getUpdateService(AdGroupType adGroupType) {
        Objects.requireNonNull(adGroupType, "adGroup");
        return Optional.ofNullable(updateServices.get(adGroupType))
                .orElseThrow(() -> new UnsupportedOperationException("Invalid AdGroupType: " + adGroupType));
    }

    public void prefetchAdGroupsBanners(int shard, Collection<AdGroup> adGroups) {
        var adGroupIds = listToSet(adGroups, AdGroup::getId);
        var adGroupsBanners =
                StreamEx.of(bannerRepository.getSafely(shard, bannerAdGroupIdFilter(adGroupIds), getClasses()))
                        .map(b -> (BannerWithSystemFields) b)
                        .groupingBy(BannerWithSystemFields::getAdGroupId);
        adGroups.forEach(adGroup ->
                adGroup.setBanners(adGroupsBanners.getOrDefault(adGroup.getId(), emptyList())));
    }

    private Collection<Class<? extends Banner>> getClasses() {
        var classes = new ArrayList<>(bannerTextExtractor.getExtractorClasses());
        classes.add(BannerWithSystemFields.class);
        return classes;
    }

    public Map<Long, List<Long>> getBannersMinusGeoByAdGroupIds(int shard, Collection<Long> adGroupIds) {
        return bannerModerationRepository.getBannersMinusGeoByAdGroupIds(shard, adGroupIds);
    }
}
