package ru.yandex.direct.core.entity.adgroup.service.update.types;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Set;
import java.util.stream.IntStream;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.DynamicAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.DynamicFeedAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.DynamicTextAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.StatusBLGenerated;
import ru.yandex.direct.core.entity.adgroup.model.StatusModerate;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.adgroup.service.ModerationMode;
import ru.yandex.direct.core.entity.adgroup.service.update.AbstractAdGroupUpdateService;
import ru.yandex.direct.core.entity.adgroup.service.update.AdGroupPostUpdateOptions;
import ru.yandex.direct.core.entity.adgroup.service.update.AdGroupUpdateData;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerModerationRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.creative.service.CreativeService;
import ru.yandex.direct.core.entity.domain.service.DomainService;
import ru.yandex.direct.core.entity.dynamictextadtarget.container.DynamicTextAdTargetSelectionCriteria;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicFeedAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.service.DynamicTextAdTargetService;
import ru.yandex.direct.core.entity.moderation.repository.sending.BannerMulticardSetSendingRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.regions.GeoTreeFactory;

import static com.google.common.base.Preconditions.checkState;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.utils.FunUtilsKt.mapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
@Service
public class DynamicUpdateService extends AbstractAdGroupUpdateService {
    private final DomainService domainService;
    private final DynamicTextAdTargetService dynamicTextAdTargetService;
    private final RbacService rbacService;

    @Autowired
    public DynamicUpdateService(
            DslContextProvider dslContextProvider,
            BannerCommonRepository bannerCommonRepository,
            BannerModerationRepository bannerModerationRepository,
            BannerMulticardSetSendingRepository bannerMulticardSetSendingRepository,
            CampaignRepository campaignRepository,
            AdGroupRepository adGroupRepository,
            CreativeService creativeService,
            GeoTreeFactory geoTreeFactory,
            DomainService domainService,
            DynamicTextAdTargetService dynamicTextAdTargetService,
            RbacService rbacService) {
        super(
                AdGroupType.DYNAMIC,
                dslContextProvider,
                bannerCommonRepository,
                bannerModerationRepository,
                bannerMulticardSetSendingRepository,
                campaignRepository,
                adGroupRepository,
                creativeService,
                geoTreeFactory);
        this.domainService = domainService;
        this.dynamicTextAdTargetService = dynamicTextAdTargetService;
        this.rbacService = rbacService;
    }

    @Override
    protected void beforeUpdateInTransaction(int shard, ClientId clientId, List<AppliedChanges<AdGroup>> adGroups) {
        super.beforeUpdateInTransaction(shard, clientId, adGroups);

        List<Integer> adGroupWithChangedDomainUrlIndexes = IntStream.range(0, adGroups.size())
                .filter(i -> adGroups.get(i).getModel() instanceof DynamicTextAdGroup
                        && adGroups.get(i).castModelUp(DynamicTextAdGroup.class).changed(DynamicTextAdGroup.DOMAIN_URL))
                .boxed()
                .collect(toList());
        if (adGroupWithChangedDomainUrlIndexes.isEmpty()) {
            return;
        }

        List<Long> newDomainUrlIds = domainService.getOrCreate(
                dslContextProvider.ppc(shard),
                adGroupWithChangedDomainUrlIndexes.stream()
                        .map(i -> adGroups.get(i).castModelUp(DynamicTextAdGroup.class).getModel().getDomainUrl())
                        .collect(toList()));

        checkState(newDomainUrlIds.size() == adGroupWithChangedDomainUrlIndexes.size());

        IntStream.range(0, adGroupWithChangedDomainUrlIndexes.size())
                .forEach(i -> adGroups.get(adGroupWithChangedDomainUrlIndexes.get(i))
                        .castModelUp(DynamicTextAdGroup.class)
                        .modify(DynamicTextAdGroup.MAIN_DOMAIN_ID, newDomainUrlIds.get(i)));

        List<AppliedChanges<AdGroup>> changesWithFeedId = filterList(adGroups, adGroupChange -> {
            if (adGroupChange.getModel() instanceof DynamicFeedAdGroup) {
                AppliedChanges<DynamicFeedAdGroup> changes = adGroupChange.castModelUp(DynamicFeedAdGroup.class);
                return changes.changed(DynamicFeedAdGroup.FEED_ID);
            }
            return false;
        });
        Set<Long> filteredAdGroupsIds = mapToSet(changesWithFeedId, ac -> ac.getModel().getId());
        deleteDynamicFeedFilters(clientId, filteredAdGroupsIds);
    }

    /**
     * См. protected/Direct/AdGroups2/Dynamic.pm::prepare_update
     */
    @Override
    protected AdGroupPostUpdateOptions doPrepareForUpdate(AdGroupUpdateData adGroupUpdateData,
                                                          ModerationMode moderationMode) {
        AdGroupPostUpdateOptions result = super.doPrepareForUpdate(adGroupUpdateData, moderationMode);

        AppliedChanges<AdGroup> adGroupChanges = adGroupUpdateData.getAdGroupChanges();

        // Поведение аналогично https://a.yandex-team.ru/arc_vcs/direct/core/src/main/java/ru/yandex/direct/core/entity/adgroup/service/update/types/PerformanceUpdateService.java?rev=r8688301#L56
        // Добавлено вместе с новыми полями
        AppliedChanges<DynamicAdGroup> changes = adGroupChanges.castModelUp(DynamicAdGroup.class);

        if (changes.changed(DynamicAdGroup.FIELD_TO_USE_AS_NAME) ||
                changes.changed(DynamicAdGroup.FIELD_TO_USE_AS_BODY)) {
            changes.modify(DynamicAdGroup.STATUS_B_L_GENERATED, StatusBLGenerated.PROCESSING);
            changes.modify(DynamicAdGroup.STATUS_BS_SYNCED, StatusBsSynced.NO);
        }

        if (changes.changed(DynamicAdGroup.RELEVANCE_MATCH_CATEGORIES)) {
            adGroupChanges.modify(AdGroup.STATUS_BS_SYNCED, StatusBsSynced.NO);
            adGroupChanges.modifyIfNotChanged(AdGroup.LAST_CHANGE, LocalDateTime::now);
        }
        if (adGroupChanges.getModel() instanceof DynamicTextAdGroup) {
            AppliedChanges<DynamicTextAdGroup> textAdGroupChanges =
                    adGroupChanges.castModelUp(DynamicTextAdGroup.class);
            DynamicTextAdGroup textAdGroup = textAdGroupChanges.getModel();

            if (textAdGroupChanges.changed(DynamicTextAdGroup.DOMAIN_URL)) {
                textAdGroupChanges.modify(AdGroup.STATUS_BS_SYNCED, StatusBsSynced.NO);
                textAdGroupChanges.modifyIfNotChanged(AdGroup.LAST_CHANGE, LocalDateTime::now);

                // Обновим статус генерации фраз в BannerLand
                if (textAdGroup.getStatusModerate() != StatusModerate.NEW
                        && textAdGroup.getStatusBLGenerated() == StatusBLGenerated.NO) {
                    textAdGroupChanges.modify(
                            DynamicTextAdGroup.STATUS_B_L_GENERATED,
                            StatusBLGenerated.PROCESSING);
                }
            }
        } else if (!(adGroupChanges.getModel() instanceof DynamicFeedAdGroup)) {
            throw new UnsupportedOperationException(
                    "Unknown dynamic adgroup class: " + adGroupChanges.getModel().getClass());
        }

        // Сброс statusBsSynced означает, что нужно также сбросить этот статус на всех баннерах в группе,
        // поскольку некоторые поля группы передаются в БК в составе баннеров
        if (adGroupChanges.getModel().getStatusBsSynced() == StatusBsSynced.NO) {
            result.setResetBannersStatusBsSync(true);
        }

        return result;
    }

    private void deleteDynamicFeedFilters(
            ClientId clientId,
            Set<Long> adGroupIds) {
        long operatorUid = rbacService.getChiefByClientId(clientId);

        List<Long> dynamicFeedAdTargetIds = mapList(dynamicTextAdTargetService.getDynamicFeedAdTargets(
                clientId, operatorUid,
                new DynamicTextAdTargetSelectionCriteria().withAdGroupIds(adGroupIds), LimitOffset.maxLimited()
        ), DynamicFeedAdTarget::getId);

        dynamicTextAdTargetService.deleteDynamicAdTargets(operatorUid, clientId, dynamicFeedAdTargetIds);
    }
}
