package ru.yandex.direct.core.entity.adgroup.service.update.types;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.PerformanceAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.StatusBLGenerated;
import ru.yandex.direct.core.entity.adgroup.model.StatusModerate;
import ru.yandex.direct.core.entity.adgroup.model.StatusPostModerate;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.adgroup.service.ModerationMode;
import ru.yandex.direct.core.entity.adgroup.service.update.AbstractAdGroupUpdateService;
import ru.yandex.direct.core.entity.adgroup.service.update.AdGroupPostUpdateOptions;
import ru.yandex.direct.core.entity.adgroup.service.update.AdGroupUpdateData;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerModerationRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.creative.service.CreativeService;
import ru.yandex.direct.core.entity.moderation.repository.sending.BannerMulticardSetSendingRepository;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilter;
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.regions.GeoTreeFactory;

import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.flatMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
@Service
public class PerformanceUpdateService extends AbstractAdGroupUpdateService {
    private final PerformanceFilterService performanceFilterService;
    private final RbacService rbacService;

    @Autowired
    public PerformanceUpdateService(
            DslContextProvider dslContextProvider,
            BannerCommonRepository bannerCommonRepository,
            BannerModerationRepository bannerModerationRepository,
            BannerMulticardSetSendingRepository bannerMulticardSetSendingRepository,
            CampaignRepository campaignRepository,
            AdGroupRepository adGroupRepository,
            CreativeService creativeService,
            GeoTreeFactory geoTreeFactory,
            PerformanceFilterService performanceFilterService,
            RbacService rbacService) {
        super(
                AdGroupType.PERFORMANCE,
                dslContextProvider,
                bannerCommonRepository,
                bannerModerationRepository,
                bannerMulticardSetSendingRepository,
                campaignRepository,
                adGroupRepository,
                creativeService,
                geoTreeFactory);
        this.performanceFilterService = performanceFilterService;
        this.rbacService = rbacService;
    }

    @Override
    protected void beforeUpdateInTransaction(int shard, ClientId clientId, List<AppliedChanges<AdGroup>> adGroups) {
        List<AppliedChanges<AdGroup>> changesWithFeedId = filterList(adGroups, adGroupChange -> {
            AppliedChanges<PerformanceAdGroup> changes = adGroupChange.castModelUp(PerformanceAdGroup.class);
            return changes.changed(PerformanceAdGroup.FEED_ID);
        });
        List<Long> filteredAdGroupsIds = mapList(changesWithFeedId, ac -> ac.getModel().getId());
        deletePerformanceFeedFilters(clientId, filteredAdGroupsIds);
    }

    /**
     * Реализуем логику, что статусы statusBsSynced и statusBlGenerated сбрасываются при изменении полей смарт-группы.
     * Подробнее тут: https://st.yandex-team.ru/DIRECT-93025#5c90cfb0a3dbb9001fe4893a
     */
    @Override
    protected AdGroupPostUpdateOptions doPrepareForUpdate(AdGroupUpdateData adGroupsUpdateData,
                                                          ModerationMode moderationMode) {
        AppliedChanges<AdGroup> adGroupChanges = adGroupsUpdateData.getAdGroupChanges();
        AppliedChanges<PerformanceAdGroup> changes = adGroupChanges.castModelUp(PerformanceAdGroup.class);
        if (changes.changed(PerformanceAdGroup.FIELD_TO_USE_AS_NAME) ||
                changes.changed(PerformanceAdGroup.FIELD_TO_USE_AS_BODY)) {
            changes.modify(PerformanceAdGroup.STATUS_B_L_GENERATED, StatusBLGenerated.PROCESSING);
            changes.modify(PerformanceAdGroup.STATUS_BS_SYNCED, StatusBsSynced.NO);
        }
        AdGroupPostUpdateOptions updateOptions = super.doPrepareForUpdate(adGroupsUpdateData, moderationMode);

        // поправим статусы модерации после дефолтного doPrepareForUpdate
        AdGroup adGroup = adGroupChanges.getModel();
        if (adGroup.getType() == AdGroupType.PERFORMANCE
                && adGroup.getStatusModerate() != StatusModerate.NEW
                && adGroup.getStatusModerate() != StatusModerate.YES
        ) {
            // performance-группы не-черновики сразу помечаем промодерированными
            // они не модерируются сами по себе
            adGroupChanges.modify(AdGroup.STATUS_MODERATE, StatusModerate.YES);
            adGroupChanges.modify(AdGroup.STATUS_POST_MODERATE, StatusPostModerate.YES);
        }

        // Сброс statusBsSynced означает, что нужно также сбросить этот статус на всех баннерах в группе,
        // поскольку некоторые поля группы передаются в БК в составе баннеров
        if (adGroupChanges.getModel().getStatusBsSynced() == StatusBsSynced.NO) {
            updateOptions.setResetBannersStatusBsSync(true);
        }

        return updateOptions;
    }

    private void deletePerformanceFeedFilters(
            ClientId clientId,
            List<Long> adGroupIds) {
        long operatorUid = rbacService.getChiefByClientId(clientId);

        List<Long> performanceFilterIds = flatMap(performanceFilterService.getPerformanceFilters(clientId,
                        adGroupIds).values(),
                adGroups -> mapList(adGroups, PerformanceFilter::getId));

        performanceFilterService.deletePerformanceFilters(clientId, operatorUid, performanceFilterIds);
    }

}
