package ru.yandex.direct.core.entity.adgroup.service.validation;

import java.util.Arrays;
import java.util.List;
import java.util.Map;

import com.google.common.collect.ImmutableMultimap;
import com.google.common.collect.ImmutableSetMultimap;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.DynamicAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.GeoproductAvailability;
import ru.yandex.direct.core.entity.campaign.model.CampaignForAccessCheck;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;
import ru.yandex.direct.core.entity.client.model.ClientLimits;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.result.Defect;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.inconsistentAdGroupType;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.inconsistentAdGroupTypeToCampaignType;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.inconsistentDynamicAdGroupType;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.maxAdGroupsInCampaign;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

public class AdGroupConstraints {

    private static final ImmutableMultimap<AdGroupType, CampaignType> ALLOWED_ADGROUP_TYPE_TO_CAMPAIGN_TYPE;

    public static final int MIN_GEO_SEGMENTS_COUNT_IN_HYPER_GEO = 1;
    public static final int MAX_GEO_SEGMENTS_COUNT_IN_HYPER_GEO = 1;
    public static final int MAX_GEO_SEGMENTS_COUNT_WITH_FEATURE_IN_HYPER_GEO = 100;

    static {
        ImmutableSetMultimap.Builder<AdGroupType, CampaignType> builder = ImmutableSetMultimap.builder();
        builder.put(AdGroupType.BASE, CampaignType.TEXT);
        builder.put(AdGroupType.MOBILE_CONTENT, CampaignType.MOBILE_CONTENT);
        builder.put(AdGroupType.PERFORMANCE, CampaignType.PERFORMANCE);
        builder.put(AdGroupType.DYNAMIC, CampaignType.DYNAMIC);
        builder.put(AdGroupType.MCBANNER, CampaignType.MCBANNER);
        builder.putAll(AdGroupType.CPM_BANNER, Arrays.asList(CampaignType.CPM_BANNER, CampaignType.CPM_DEALS, CampaignType.CPM_PRICE));
        builder.put(AdGroupType.CPM_GEOPRODUCT, CampaignType.CPM_BANNER);
        builder.put(AdGroupType.CPM_GEO_PIN, CampaignType.CPM_BANNER);
        builder.putAll(AdGroupType.CPM_AUDIO, Arrays.asList(CampaignType.CPM_BANNER, CampaignType.CPM_PRICE));
        builder.putAll(AdGroupType.CPM_VIDEO, Arrays.asList(CampaignType.CPM_BANNER, CampaignType.CPM_PRICE));
        builder.put(AdGroupType.CPM_OUTDOOR, CampaignType.CPM_BANNER);
        builder.put(AdGroupType.CPM_INDOOR, CampaignType.CPM_BANNER);
        builder.putAll(AdGroupType.CPM_YNDX_FRONTPAGE, Arrays.asList(CampaignType.CPM_YNDX_FRONTPAGE, CampaignType.CPM_PRICE));
        builder.put(AdGroupType.CONTENT_PROMOTION_VIDEO, CampaignType.CONTENT_PROMOTION);
        builder.put(AdGroupType.CONTENT_PROMOTION, CampaignType.CONTENT_PROMOTION);
        builder.put(AdGroupType.INTERNAL, CampaignType.INTERNAL_FREE);
        builder.put(AdGroupType.INTERNAL, CampaignType.INTERNAL_DISTRIB);
        builder.put(AdGroupType.INTERNAL, CampaignType.INTERNAL_AUTOBUDGET);
        ALLOWED_ADGROUP_TYPE_TO_CAMPAIGN_TYPE = builder.build();
    }

    private AdGroupConstraints() {
        // no instantiation
    }

    static Constraint<AdGroup, Defect> maxAdGroupCountInCampaign(Long adGroupCount,
                                                                 ClientLimits clientLimits) {
        return fromPredicate(adgroup -> adGroupCount <= clientLimits.getBannersCountLimitOrDefault(),
                maxAdGroupsInCampaign(clientLimits.getBannersCountLimitOrDefault()));
    }

    static Constraint<AdGroupType, Defect> isAdGroupTypeCorrespondTo(CampaignForAccessCheck campaign) {
        return fromPredicate(adGroupType -> campaign != null && ALLOWED_ADGROUP_TYPE_TO_CAMPAIGN_TYPE
                        .containsEntry(adGroupType, campaign.getType()),
                inconsistentAdGroupTypeToCampaignType());
    }

    /**
     * Проверка совместимости типов групп в рамках текущей кампании.
     *
     * @param anyGeoproduct наличие группы типа {@link AdGroupType#CPM_GEOPRODUCT} или {@link AdGroupType#CPM_GEO_PIN} в текущей кампании.
     * @return ошибка, если:
     * <ol>
     * <li>при добавлении групп {@link AdGroupType#CPM_GEOPRODUCT} или {@link AdGroupType#CPM_GEO_PIN} в кампании имеются только другие типы групп;</li>
     * <li>при добавлении групп не {@link AdGroupType#CPM_GEOPRODUCT} и не {@link AdGroupType#CPM_GEO_PIN} в кампании уже имеется тип {@link AdGroupType#CPM_GEOPRODUCT} или {@link AdGroupType#CPM_GEO_PIN}.</li>
     * </ol>
     */
    public static Constraint<AdGroupType, Defect> isAdGroupTypeSupportedWithGeoproduct(
            GeoproductAvailability anyGeoproduct) {
        return fromPredicate(type -> {
                    if (anyGeoproduct == GeoproductAvailability.EMPTY) {
                        return true;
                    }
                    return List.of(AdGroupType.CPM_GEOPRODUCT, AdGroupType.CPM_GEO_PIN).contains(type)
                            ? anyGeoproduct == GeoproductAvailability.YES
                            : anyGeoproduct == GeoproductAvailability.NO;
                },
                AdGroupDefects.adGroupTypeNotSupported());
    }

    static Constraint<ModelChanges<AdGroup>, Defect> hasValidTypeForUpdate(Map<Long, AdGroup> models) {
        return mc -> {
            AdGroup adGroup = models.get(mc.getId());
            checkState(adGroup != null);

            // Тип модели в ModelChanges должен быть родительским по отношению к типу модели.
            // В таком случае изменения из ModelChanges будут корректно применены к модели
            if (!mc.getModelType().isAssignableFrom(adGroup.getClass())) {
                if (DynamicAdGroup.class.isAssignableFrom(mc.getModelType()) && adGroup instanceof DynamicAdGroup) {
                    return inconsistentDynamicAdGroupType();
                } else {
                    return inconsistentAdGroupType();
                }
            }

            return null;
        };
    }

    public static Constraint<AdGroupType, Defect> isValidTypeForCpvStrategies(Long campaignId,
            Map<Long, StrategyName> strategyNameMap)
    {
        return fromPredicate(
                type -> type == AdGroupType.CPM_VIDEO
                        || strategyNameMap.get(campaignId) != StrategyName.AUTOBUDGET_AVG_CPV_CUSTOM_PERIOD
                        && strategyNameMap.get(campaignId) != StrategyName.AUTOBUDGET_AVG_CPV,
                AdGroupDefects.adGroupTypeNotSupported());
    }

    public static Constraint<AdGroupType, Defect> isValidTypeForCurrency(CurrencyCode currencyCode) {
        return fromPredicate(
                type -> type != AdGroupType.CPM_AUDIO || currencyCode != CurrencyCode.GBP,
                AdGroupDefects.adGroupTypeNotSupported());
    }
}
