package ru.yandex.direct.core.entity.adgroup.service.validation;

import java.util.List;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.service.text.BannerTextExtractor;
import ru.yandex.direct.core.entity.region.AllowedRegionsForLanguageContainer;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.queryrec.QueryrecService;
import ru.yandex.direct.queryrec.model.Language;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.GeoTreeFactory;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.result.Defect;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.inconsistentGeoWithBannerLanguages;
import static ru.yandex.direct.core.entity.banner.type.language.BannerLanguageConverter.convertLanguage;
import static ru.yandex.direct.core.entity.region.RegionConstants.LANGUAGE_TO_ALLOWED_REGIONS_CONTAINER;

/**
 * Проверяет соотвествие языка группы объявлений новому гео
 */
@Component
@ParametersAreNonnullByDefault
public class AdGroupLanguageGeoValidator {
    private final BannerTextExtractor bannerTextExtractor;
    private final QueryrecService queryrecService;

    private final GeoTreeFactory geoTreeFactory;

    @Autowired
    public AdGroupLanguageGeoValidator(
            BannerTextExtractor bannerTextExtractor,
            QueryrecService queryrecService,
            GeoTreeFactory geoTreeFactory) {
        this.bannerTextExtractor = bannerTextExtractor;
        this.queryrecService = queryrecService;
        this.geoTreeFactory = geoTreeFactory;
    }

    /**
     * Создать ограничение для проверки соотвествия языка группы объявлений новому geo
     *
     * @param adGroupsWithChangedGeoIds Идентификаторы групп объявлений с измененнным Geo
     * @param adGroupsLangByCampaign    Язык группы объявлений определенный на основе кампании
     */
    public Constraint<AdGroup, Defect> createConstraint(
            Set<Long> adGroupsWithChangedGeoIds, Function<Long, Language> adGroupsLangByCampaign, ClientId clientId,
            Long clientRegionId) {
        return adGroup -> {
            if (!adGroupsWithChangedGeoIds.contains(adGroup.getId())) {
                return null;
            }

            checkState(adGroup.getBanners() != null);

            var banners = adGroup.getBanners();

            return checkGeoConsistentWithBannersLanguage(
                    adGroupsLangByCampaign.apply(adGroup.getId()), adGroup.getGeo(), banners, clientId, clientRegionId);
        };
    }

    private Defect checkGeoConsistentWithBannersLanguage(
            Language adGroupLangByCampaign,
            List<Long> geoIds,
            List<BannerWithSystemFields> banners,
            ClientId clientId,
            Long clientRegionId) {
        var bannerToExtractedText = bannerTextExtractor.extractTexts(banners);

        for (BannerWithSystemFields banner : banners) {
            // Если язык явно задан для всех объявлений кампании, то используем его
            // в противном случае пытаемся определить язык по тексту объявления
            // Язык на кампании можется задаваться для некоторых кампаний с объявлениями
            // на турецком языке, так как на основе текста объявлений он может определяться
            // неправильно
            Language language = adGroupLangByCampaign != Language.UNKNOWN
                    ? adGroupLangByCampaign
                    : getOrDetectBannerLanguage(clientId, clientRegionId,
                    convertLanguage(banner.getLanguage()), bannerToExtractedText.get(banner));

            Set<Long> allowableGeoIds = LANGUAGE_TO_ALLOWED_REGIONS_CONTAINER
                    .getOrDefault(language, AllowedRegionsForLanguageContainer.EMPTY).getAllowedRegionIds();
            if (allowableGeoIds != null
                    && !getRussianGeoTree().isRegionsIncludedIn(geoIds, allowableGeoIds)) {
                return inconsistentGeoWithBannerLanguages(language, banner.getId());
            }
        }

        return null;
    }

    private Language getOrDetectBannerLanguage(ClientId clientId,
                                               Long clientRegionId,
                                               Language language,
                                               @Nullable String text) {
        if (language != Language.UNKNOWN) {
            return language;
        }

        Language result = queryrecService.recognize(text, clientId, clientRegionId);

        if (result == null || result == Language.UNKNOWN) {
            result = Language.RUSSIAN;
        }

        return result;
    }

    private GeoTree getRussianGeoTree() {
        // Для валидации соответствия регионов с языком кампании и баннеров используется российское геодерево
        return geoTreeFactory.getRussianGeoTree();
    }
}
