package ru.yandex.direct.core.entity.adgroup.service.validation;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.BiFunction;

import one.util.streamex.EntryStream;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.campaign.AvailableCampaignSources;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeSource;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessConstraint;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignAccessType;
import ru.yandex.direct.core.entity.hypergeo.model.HyperGeo;
import ru.yandex.direct.core.entity.region.validation.RegionIdsValidator;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.function.Predicate.not;
import static ru.yandex.direct.common.util.TextUtils.smartStrip;
import static ru.yandex.direct.core.entity.adgroup.service.AdGroupCpmPriceUtils.isDefaultAdGroup;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupConstraints.MAX_GEO_SEGMENTS_COUNT_IN_HYPER_GEO;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupConstraints.MAX_GEO_SEGMENTS_COUNT_WITH_FEATURE_IN_HYPER_GEO;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupConstraints.MIN_GEO_SEGMENTS_COUNT_IN_HYPER_GEO;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.adGroupNameCantBeEmpty;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.adGroupNameIsNotSet;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.duplicatedObject;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.notFound;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.unableToDelete;
import static ru.yandex.direct.core.entity.hypergeo.validation.HyperGeoDefectsKt.hyperGeoSegmentsSizeInInterval;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseConstraints.MAX_LINKED_PACKS_TO_ONE_AD_GROUP;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseDefects.minusWordsPackNotFound;
import static ru.yandex.direct.core.validation.defects.Defects.badStatusCampaignArchived;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.maxListSize;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notInSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.constraint.StringConstraints.matchPattern;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;
import static ru.yandex.direct.validation.constraint.StringConstraints.notEmpty;
import static ru.yandex.direct.validation.constraint.StringConstraints.onlyUtf8Mb3Symbols;
import static ru.yandex.direct.validation.defect.CommonDefects.objectNotFound;

@Service
public class AdGroupValidationService {
    public static final int MAX_NAME_LENGTH = 255;
    static final int MAX_TAGS_COUNT = 30;
    static final int MAX_PAGE_GROUP_TAGS_COUNT = 50;
    public static final int MAX_TARGET_TAGS_COUNT = 50;
    public static final int MAX_TAG_LENGTH = 50;
    public static final String TAG_PATTERN = "^[a-zA-Z0-9_\\-]+\\z";

    private final RegionIdsValidator regionIdsValidator;
    private final AdGroupRepository adGroupRepository;
    private final CampaignRepository campaignRepository;
    private final CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory;

    @Autowired
    public AdGroupValidationService(
            RegionIdsValidator regionIdsValidator,
            AdGroupRepository adGroupRepository,
            CampaignRepository campaignRepository,
            CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory) {
        this.regionIdsValidator = regionIdsValidator;
        this.adGroupRepository = adGroupRepository;
        this.campaignRepository = campaignRepository;
        this.campaignSubObjectAccessCheckerFactory = campaignSubObjectAccessCheckerFactory;
    }

    public ValidationResult<AdGroup, Defect> validateAdGroup(
            AdGroup adGroup,
            GeoTree geoTree,
            BiFunction<AdGroup, ModelProperty, Boolean> isPropertyChanged,
            Map<Long, List<Long>> campaignsTags,
            Set<Long> existingLibraryMinusWordIds,
            Map<Long, HyperGeo> existingHyperGeoById,
            boolean hyperlocalIsEnabled,
            boolean multipleSegmentsInHyperGeo) {
        ModelItemValidationBuilder<AdGroup> vb = ModelItemValidationBuilder.of(adGroup);

        if (isPropertyChanged.apply(adGroup, AdGroup.NAME)) {
            vb.item(smartStrip(adGroup.getName()), AdGroup.NAME.name())
                    .check(notNull(), adGroupNameIsNotSet())
                    .check(notEmpty(), adGroupNameCantBeEmpty())
                    .check(maxStringLength(MAX_NAME_LENGTH))
                    .check(onlyUtf8Mb3Symbols());
        }

        boolean isHyperLocalAdGroup = adGroup.getHyperGeoId() != null && hyperlocalIsEnabled;

        // Для гиперлокальных групп всегда проставляем покрывающее гео для геосегментов, поэтому валидировать то,
        // что пришло извне операции нет смысла
        if (isPropertyChanged.apply(adGroup, AdGroup.GEO) && !isHyperLocalAdGroup) {
            vb.item(AdGroup.GEO)
                    .check(notNull())
                    .checkBy(geo -> regionIdsValidator.apply(geo, geoTree), When.notNull());
        }

        int maxGeoSegmentsCount = multipleSegmentsInHyperGeo
                ? MAX_GEO_SEGMENTS_COUNT_WITH_FEATURE_IN_HYPER_GEO
                : MAX_GEO_SEGMENTS_COUNT_IN_HYPER_GEO;

        if (isPropertyChanged.apply(adGroup, AdGroup.HYPER_GEO_ID)) {
            vb.item(AdGroup.HYPER_GEO_ID)
                    .check(isNull(), When.isFalse(hyperlocalIsEnabled))
                    .check(validId(), When.isValid())
                    .check(fromPredicate(existingHyperGeoById::containsKey, objectNotFound()), When.isValid())
                    .check(fromPredicate(hyperGeoId -> {
                                var hyperGeo = existingHyperGeoById.get(hyperGeoId);
                                var hyperGeoSegmentSize = hyperGeo != null ? hyperGeo.getHyperGeoSegments().size() : 0;
                                return hyperGeoSegmentSize <= maxGeoSegmentsCount;
                            },
                            hyperGeoSegmentsSizeInInterval(MIN_GEO_SEGMENTS_COUNT_IN_HYPER_GEO, maxGeoSegmentsCount)),
                            When.isValid());
        }

        if (isPropertyChanged.apply(adGroup, AdGroup.TAGS) && adGroup.getTags() != null) {
            Set<Long> existingTags = Optional.ofNullable(campaignsTags.get(adGroup.getCampaignId()))
                    .map(HashSet::new)
                    .orElse(new HashSet<>());
            vb.list(AdGroup.TAGS)
                    .checkEach(notNull())
                    .checkEach(unique(), When.isValid())
                    .checkEach(inSet(existingTags), objectNotFound(), When.isValid())
                    .check(maxListSize(MAX_TAGS_COUNT), When.isValid());
        }

        if (isPropertyChanged.apply(adGroup, AdGroup.PAGE_GROUP_TAGS) && adGroup.getPageGroupTags() != null) {
            vb.list(AdGroup.PAGE_GROUP_TAGS)
                    .checkEach(notNull())
                    .checkEach(maxStringLength(MAX_TAG_LENGTH), When.isValid())
                    .checkEach(unique(), When.isValid())
                    .checkEach(matchPattern(TAG_PATTERN), When.isValid())
                    .check(maxListSize(MAX_PAGE_GROUP_TAGS_COUNT), When.isValid());
        }

        if (isPropertyChanged.apply(adGroup, AdGroup.TARGET_TAGS) && adGroup.getTargetTags() != null) {
            vb.list(AdGroup.TARGET_TAGS)
                    .checkEach(notNull())
                    .checkEach(maxStringLength(MAX_TAG_LENGTH), When.isValid())
                    .checkEach(unique(), When.isValid())
                    .checkEach(matchPattern(TAG_PATTERN), When.isValid())
                    .check(maxListSize(MAX_TARGET_TAGS_COUNT), When.isValid());
        }

        if (isPropertyChanged.apply(adGroup, AdGroup.LIBRARY_MINUS_KEYWORDS_IDS)) {
            vb.list(AdGroup.LIBRARY_MINUS_KEYWORDS_IDS)
                    .checkEach(notNull())
                    .checkEach(unique(), When.isValid())
                    .checkEach(inSet(existingLibraryMinusWordIds), minusWordsPackNotFound(), When.isValid())
                    .check(maxListSize(MAX_LINKED_PACKS_TO_ONE_AD_GROUP), When.isValid());
        }

        if (isPropertyChanged.apply(adGroup, AdGroup.CONTENT_CATEGORIES_RETARGETING_CONDITION_RULES)) {
            vb.list(AdGroup.CONTENT_CATEGORIES_RETARGETING_CONDITION_RULES)
                    .checkBy(ContentCategoriesRetargetingConditionRulesValidator::validate);
        }

        return vb.getResult();
    }

    public ValidationResult<List<Long>, Defect> validateDelete(int shard, Long operatorUid, ClientId clientId,
                                                               List<Long> adGroupIds) {

        CampaignSubObjectAccessConstraint accessConstraint = campaignSubObjectAccessCheckerFactory
                .newAdGroupChecker(operatorUid, clientId, adGroupIds)
                .createAdGroupValidator(CampaignAccessType.READ_WRITE)
                .getAccessConstraint();

        Set<Long> existingAdGroups = adGroupRepository.getClientExistingAdGroupIds(shard, clientId, adGroupIds);
        Set<Long> adGroupsWithConditions = adGroupRepository.getAdGroupIdsWithAnyConditions(shard, adGroupIds);
        Set<Long> adGroupsWithAds = adGroupRepository.getAdGroupIdsWithBanners(shard, adGroupIds);
        Set<Long> adGroupsInArchivedCampaigns = adGroupRepository.getAdGroupIdsInArchivedCampaigns(shard, adGroupIds);

        Map<Long, CampaignWithType> campaignWithTypeByAdGroupId = campaignRepository
                .getCampaignsWithTypeByAdGroupIds(shard, clientId, existingAdGroups);
        List<Long> cpmPriceAdGroups = EntryStream.of(campaignWithTypeByAdGroupId)
                .mapValues(CampaignWithType::getType)
                .filterValues(CampaignType.CPM_PRICE::equals)
                .keys()
                .toList();
        Map<Long, Long> cpmPriceAdGroupPriorities = adGroupRepository.getAdGroupsPriority(shard, cpmPriceAdGroups);

        return ListValidationBuilder.<Long, Defect>of(adGroupIds)
                .check(notNull())
                .checkEach(notNull())
                .checkEach(validId(), When.isValid())
                .checkEach(unique(), duplicatedObject(), When.isValid())
                .checkEach(inSet(existingAdGroups), notFound(), When.isValid())
                .checkEach(accessConstraint, When.isValid())
                .checkEach(notInSet(adGroupsInArchivedCampaigns), badStatusCampaignArchived(), When.isValid())
                .checkEach(notInSet(adGroupsWithConditions), unableToDelete(), When.isValid())
                .checkEach(notInSet(adGroupsWithAds), unableToDelete(), When.isValid())
                .checkEach(fromPredicate(not(isDefaultAdGroup(cpmPriceAdGroupPriorities)), unableToDelete()),
                        When.isValid())
                .getResult();
    }

    /**
     * Валидация группы
     *
     * @param adGroup                             группа
     * @param campaignTypeSourceById              маппа id кампании -> тип-источник
     * @param multipleSegmentsForUcCampaigns      гиперлокальность в UC с добавлением нескольких сегментов/кругов
     * @param hyperGeoForSmartAndDynamicCampaigns гиперлокальность в Смартах и ДО с добавлением нескольких
     *                                            сегментов/кругов
     * @param validator                           функция валидирования группы
     */
    public ValidationResult<AdGroup, Defect> validateAdGroup(
            AdGroup adGroup,
            Map<Long, CampaignTypeSource> campaignTypeSourceById,
            boolean multipleSegmentsForUcCampaigns,
            boolean hyperGeoForSmartAndDynamicCampaigns,
            BiFunction<Boolean, Boolean, ValidationResult<AdGroup, Defect>> validator) {

        var campaignTypeSource = campaignTypeSourceById.get(adGroup.getCampaignId());
        if (campaignTypeSource == null) {
            return validator.apply(false, false);
        }

        boolean isUc = AvailableCampaignSources.INSTANCE.isUC(campaignTypeSource.getCampaignsSource());

        // All UC/UAC except RMP
        boolean isNonMobileUac = isUc && !CampaignType.MOBILE_CONTENT.equals(campaignTypeSource.getCampaignType());

        boolean hyperlocalForTextCampaignsEnabled = !isUc
                && CampaignType.TEXT.equals(campaignTypeSource.getCampaignType());
        boolean hyperlocalForSmartCampaignsEnabled = !isUc
                && CampaignType.PERFORMANCE.equals(campaignTypeSource.getCampaignType())
                && hyperGeoForSmartAndDynamicCampaigns;
        boolean hyperlocalForDynamicCampaignsEnabled = !isUc
                && CampaignType.DYNAMIC.equals(campaignTypeSource.getCampaignType())
                && hyperGeoForSmartAndDynamicCampaigns;

        boolean isHyperGeoEnabledForCampaignType = hyperlocalForTextCampaignsEnabled
                || hyperlocalForSmartCampaignsEnabled
                || hyperlocalForDynamicCampaignsEnabled;

        boolean isHyperGeoEnabled = isNonMobileUac
                || isHyperGeoEnabledForCampaignType;

        boolean isMultipleSegmentsEnabled = (isNonMobileUac && multipleSegmentsForUcCampaigns)
                || isHyperGeoEnabledForCampaignType;

        return validator.apply(isHyperGeoEnabled, isMultipleSegmentsEnabled);
    }
}
