package ru.yandex.direct.core.entity.adgroup.service.validation;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.BiFunction;

import com.google.common.collect.Iterables;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.GeoproductAvailability;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.adgroup.service.bstags.AdGroupBsTagsSettings;
import ru.yandex.direct.core.entity.adgroup.service.bstags.AllowedBsTagsValidator;
import ru.yandex.direct.core.entity.adgroup.service.geotree.AdGroupGeoTreeProvider;
import ru.yandex.direct.core.entity.adgroup.service.validation.types.AdGroupTypeSpecificValidationProvider;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignForAccessCheck;
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeSource;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessConstraint;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignAccessType;
import ru.yandex.direct.core.entity.client.model.ClientLimits;
import ru.yandex.direct.core.entity.client.service.ClientLimitsService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.hypergeo.model.HyperGeo;
import ru.yandex.direct.core.entity.minuskeywordspack.repository.MinusKeywordsPackRepository;
import ru.yandex.direct.core.entity.tag.repository.TagRepository;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupConstraints.isAdGroupTypeCorrespondTo;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupConstraints.isAdGroupTypeSupportedWithGeoproduct;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupConstraints.isValidTypeForCpvStrategies;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupConstraints.isValidTypeForCurrency;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupConstraints.maxAdGroupCountInCampaign;
import static ru.yandex.direct.feature.FeatureName.HYPERLOCAL_GEO_FOR_UC_CAMPAIGNS_ENABLED_FOR_DNA;
import static ru.yandex.direct.feature.FeatureName.HYPERLOCAL_GEO_IN_SMART_AND_DO_FOR_DNA;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.nullSafetyFlatMap;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;

@Service
public class AddAdGroupValidationService {

    private final AdGroupValidationService adGroupValidationService;
    private final CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory;
    private final AdGroupRepository adGroupRepository;
    private final TagRepository tagRepository;
    private final MinusKeywordsPackRepository packRepository;
    private final ClientLimitsService clientLimitsService;
    private final FeatureService featureService;
    private final AdGroupTypeSpecificValidationProvider validationProvider;
    private final CampaignService campaignService;
    private final ClientService clientService;

    @Autowired
    public AddAdGroupValidationService(
            AdGroupValidationService adGroupValidationService,
            CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory,
            AdGroupRepository adGroupRepository,
            TagRepository tagRepository,
            MinusKeywordsPackRepository packRepository,
            ClientLimitsService clientLimitsService,
            FeatureService featureService,
            AdGroupTypeSpecificValidationProvider validationProvider,
            CampaignService campaignService,
            ClientService clientService) {
        this.adGroupValidationService = adGroupValidationService;
        this.campaignSubObjectAccessCheckerFactory = campaignSubObjectAccessCheckerFactory;
        this.adGroupRepository = adGroupRepository;
        this.tagRepository = tagRepository;
        this.packRepository = packRepository;
        this.clientLimitsService = clientLimitsService;
        this.featureService = featureService;
        this.validationProvider = validationProvider;
        this.campaignService = campaignService;
        this.clientService = clientService;
    }

    public ValidationResult<List<AdGroup>, Defect> preValidate(
            List<AdGroup> adGroups,
            long operatorUid,
            ClientId clientId,
            int shard) {
        Set<Long> adGroupCampaignIds = StreamEx.of(adGroups).map(AdGroup::getCampaignId).toSet();

        CampaignSubObjectAccessConstraint campaignAccessConstraint = campaignSubObjectAccessCheckerFactory
                .newCampaignChecker(operatorUid, clientId, adGroupCampaignIds)
                .createValidator(CampaignAccessType.READ_WRITE)
                .getAccessConstraint();

        var currencyCode = clientService.getWorkCurrency(clientId).getCode();

        Map<Long, StrategyName> strategyNameMap = listToMap(campaignService
                        .getCampaignsWithStrategies(clientId, adGroupCampaignIds),
                Campaign::getId,
                c -> c.getStrategy().getStrategyName());

        Map<Long, GeoproductAvailability> anyGeoproductByCampaignId =
                adGroupRepository.getGeoproductAvailabilityByCampaignId(shard, adGroupCampaignIds);
        return ListValidationBuilder.of(adGroups, Defect.class)
                .checkEachBy(adGroup ->
                        preValidateAdGroup(adGroup, campaignAccessConstraint, anyGeoproductByCampaignId,
                                strategyNameMap, currencyCode))
                .checkBy(list -> validationProvider.validateAddAdGroups(clientId, list))
                .getResult();
    }

    public ValidationResult<List<AdGroup>, Defect> validate(
            List<AdGroup> adGroups, Map<Long, HyperGeo> hyperGeoById,
            Map<Long, CampaignTypeSource> campaignTypeSourceById,
            Map<AdGroup, AdGroupBsTagsSettings> adGroupsBsTagsSettings,
            AdGroupGeoTreeProvider geoTreeProvider, ClientId clientId, Long operatorUid, int shard) {
        Set<Long> adGroupCampaignIds = StreamEx.of(adGroups).map(AdGroup::getCampaignId).toSet();

        ClientLimits newClientLimits = new ClientLimits();
        newClientLimits.withBannersCountLimit(0L);

        ClientLimits clientLimits = Iterables.getFirst(clientLimitsService.massGetClientLimits(singletonList(clientId)),
                newClientLimits);
        Map<Long, Long> campaignIdToAdGroupCount =
                adGroupRepository.getAdGroupCountByCampaignIds(shard, adGroupCampaignIds);

        Set<Long> allTags = nullSafetyFlatMap(adGroups, AdGroup::getTags, toSet());
        Map<Long, List<Long>> existingCampaignsTags = tagRepository.getCampaignsTagIds(shard, clientId, allTags);

        Set<Long> allLibPackIds = nullSafetyFlatMap(adGroups, AdGroup::getLibraryMinusKeywordsIds, toSet());
        Set<Long> existingLibPackIds =
                packRepository.getClientExistingLibraryMinusKeywordsPackIds(shard, clientId, allLibPackIds);

        boolean hyperlocalGeoForUcCampaigns = featureService
                .isEnabledForClientId(clientId, HYPERLOCAL_GEO_FOR_UC_CAMPAIGNS_ENABLED_FOR_DNA);
        boolean hyperGeoForSmartAndDynamicCampaigns = featureService
                .isEnabledForClientId(clientId, HYPERLOCAL_GEO_IN_SMART_AND_DO_FOR_DNA);

        var allowedBsTagsValidator = new AllowedBsTagsValidator(adGroupsBsTagsSettings, featureService, clientId,
                operatorUid);
        Validator<AdGroup, Defect> validator =
                adGroup -> adGroupValidationService.validateAdGroup(
                        adGroup,
                        campaignTypeSourceById,
                        hyperlocalGeoForUcCampaigns,
                        hyperGeoForSmartAndDynamicCampaigns,
                        (hyperlocalIsEnabled, multipleSegmentsInHyperGeo) ->
                                validateAdGroup(adGroup,
                                        clientLimits,
                                        geoTreeProvider.getGeoTree(adGroup),
                                        campaignIdToAdGroupCount,
                                        existingCampaignsTags,
                                        existingLibPackIds,
                                        hyperGeoById,
                                        hyperlocalIsEnabled,
                                        multipleSegmentsInHyperGeo
                                )
                );

        return ListValidationBuilder.of(adGroups, Defect.class)
                .checkBy(new ContentPromotionContentTypesValidator(shard, adGroupRepository))
                .checkBy(list -> validationProvider.validateAdGroups(clientId, list))
                .checkEachBy(allowedBsTagsValidator, When.isValid())
                .checkEachBy(validator, When.isValid())
                .getResult();
    }

    private ValidationResult<AdGroup, Defect> preValidateAdGroup(
            AdGroup adGroup,
            CampaignSubObjectAccessConstraint campaignAccessConstraint,
            Map<Long, GeoproductAvailability> geoproductAvailabilityByCampaignId,
            Map<Long, StrategyName> strategyNameMap,
            CurrencyCode currency) {
        ModelItemValidationBuilder<AdGroup> vb = ModelItemValidationBuilder.of(adGroup);
        vb.item(AdGroup.CAMPAIGN_ID)
                .check(notNull())
                .check(validId(), When.isValid())
                .check(campaignAccessConstraint, When.isValid());

        // Если есть какие-то проблемы с существованием или доступностью кампании,
        // то остальное валидировать бессмысленно
        if (vb.getResult().hasAnyErrors()) {
            return vb.getResult();
        }

        // Пользуемся тем, что в качестве объектов к котором проверяется доступ в campaignAccessValidator
        // выступают кампании
        Map<Long, CampaignForAccessCheck> campaignIdToCampaign =
                campaignAccessConstraint.getChecker().getSubObjectIdToCamp();

        GeoproductAvailability anyGeoproduct = geoproductAvailabilityByCampaignId.get(adGroup.getCampaignId());
        vb.item(AdGroup.TYPE)
                .check(notNull(), When.isValid())
                .check(isValidTypeForCurrency(currency), When.isValid())
                .check(isAdGroupTypeCorrespondTo(campaignIdToCampaign.get(adGroup.getCampaignId())), When.isValid())
                .check(isAdGroupTypeSupportedWithGeoproduct(anyGeoproduct), When.isValid())
                .check(isValidTypeForCpvStrategies(adGroup.getCampaignId(), strategyNameMap), When.isValid());

        vb.item(AdGroup.HYPER_GEO_ID)
                .check(validId());

        return vb.getResult();
    }

    private ValidationResult<AdGroup, Defect> validateAdGroup(
            AdGroup adGroup,
            ClientLimits clientLimits,
            GeoTree geoTree,
            Map<Long, Long> campaignIdToAdGroupCount,
            Map<Long, List<Long>> existingCampaignsTags, Set<Long> existingLibPackIds,
            Map<Long, HyperGeo> hyperGeoById,
            boolean hyperlocalIsEnabled,
            boolean multipleSegmentsInHyperGeo) {
        BiFunction<AdGroup, ModelProperty, Boolean> isPropertyChanged = (ag, mp) -> true;

        ModelItemValidationBuilder<AdGroup> vb = ModelItemValidationBuilder.of(adGroup);

        vb.checkBy(ag -> adGroupValidationService
                .validateAdGroup(ag, geoTree, isPropertyChanged, existingCampaignsTags, existingLibPackIds,
                        hyperGeoById, hyperlocalIsEnabled, multipleSegmentsInHyperGeo));

        Long adGroupsCount = campaignIdToAdGroupCount.get(adGroup.getCampaignId());
        if (adGroupsCount != null) {
            campaignIdToAdGroupCount.put(adGroup.getCampaignId(), ++adGroupsCount);
            vb.check(maxAdGroupCountInCampaign(adGroupsCount, clientLimits));
        }

        return vb.getResult();
    }
}
