package ru.yandex.direct.core.entity.adgroup.service.validation;

import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalBase;
import ru.yandex.direct.core.entity.retargeting.model.GoalType;
import ru.yandex.direct.core.entity.retargeting.model.Rule;
import ru.yandex.direct.core.entity.retargeting.model.RuleType;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.duplicatedObject;
import static ru.yandex.direct.core.entity.retargeting.Constants.MAX_GOALS_PER_RULE;
import static ru.yandex.direct.core.entity.retargeting.Constants.MAX_RULES_PER_CONDITION;
import static ru.yandex.direct.core.entity.retargeting.Constants.MIN_GOALS_PER_RULE;
import static ru.yandex.direct.core.entity.retargeting.Constants.MIN_RULES_PER_CONDITION;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.unsupportedGoalId;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.collectionSize;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isEqual;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;

@ParametersAreNonnullByDefault
public final class ContentCategoriesRetargetingConditionRulesValidator {

    public static final Set<GoalType> ALLOWED_GOAL_TYPES = Set.of(GoalType.CONTENT_CATEGORY, GoalType.CONTENT_GENRE);

    private ContentCategoriesRetargetingConditionRulesValidator() {
    }

    public static ValidationResult<List<Rule>, Defect> validate(List<Rule> rules) {
        return ListValidationBuilder.of(rules, Defect.class)
                .check(collectionSize(MIN_RULES_PER_CONDITION, MAX_RULES_PER_CONDITION))
                .checkEach(notNull())
                .checkEachBy(ContentCategoriesRetargetingConditionRulesValidator::validateRule, When.isValid())
                .getResult();
    }

    private static ValidationResult<Rule, Defect> validateRule(Rule rule) {
        var vb = ModelItemValidationBuilder.of(rule);
        vb.item(Rule.GOALS)
                .check(notNull())
                .check(collectionSize(MIN_GOALS_PER_RULE, MAX_GOALS_PER_RULE));
        vb.list(Rule.GOALS)
                .checkEach(notNull())
                .checkEach(unique(GoalBase::getId), duplicatedObject(), When.isValid())
                .checkEachBy(ContentCategoriesRetargetingConditionRulesValidator::validateGoal, When.isValid());
        vb.item(Rule.TYPE)
                .check(notNull())
                .check(isEqual(RuleType.OR, invalidValue()));
        return vb.getResult();
    }

    private static ValidationResult<Goal, Defect> validateGoal(Goal goal) {
        var vb = ModelItemValidationBuilder.of(goal);

        vb.item(Goal.ID)
                .check(notNull())
                .check(validId())
                .check(Constraint.fromPredicate(
                        goalId -> ALLOWED_GOAL_TYPES.contains(Goal.computeType(goalId)), unsupportedGoalId()),
                        When.isValid());

        return vb.getResult();
    }
}
