package ru.yandex.direct.core.entity.adgroup.service.validation;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.ContentPromotionAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.ContentPromotionAdgroupType;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.DefaultValidator;

import static java.util.Collections.emptySet;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.contentPromotionDistinctTypesWithExisting;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.contentPromotionSeveralTypesNotAllowed;

//Проверка, что в переданном списке групп не более одного типа контента
//Так как группы тип продвижения менять не могут, проверка только при добавлении групп
public class ContentPromotionContentTypesValidator implements DefaultValidator<List<AdGroup>> {
    private static final Set<AdGroupType> CONTENT_PROMOTION_ADGROUPS_TYPES = Set.of(AdGroupType.CONTENT_PROMOTION,
            AdGroupType.CONTENT_PROMOTION_VIDEO);

    private int shard;
    private final AdGroupRepository adGroupRepository;

    public ContentPromotionContentTypesValidator(int shard,
                                                 AdGroupRepository adGroupRepository) {
        this.shard = shard;
        this.adGroupRepository = adGroupRepository;
    }


    @Override
    public ValidationResult<List<AdGroup>, Defect> apply(List<AdGroup> adGroups) {
        ListValidationBuilder<AdGroup, Defect> lvb = ListValidationBuilder.of(adGroups);

        //посчитаем набор типов продвижения по cid-ам из запроса
        //Затем провалидируем число различных типов продвижения для групп одной кампании внутри запроса
        Map<Long, Set<ContentPromotionAdgroupType>> calcContentPromotionRequestAdGroupTypesByCid = StreamEx.of(adGroups)
                .filter(adGroup -> CONTENT_PROMOTION_ADGROUPS_TYPES.contains(adGroup.getType()))
                .mapToEntry(AdGroup::getCampaignId, adGroup -> calcContentPromotionAdgroupType(adGroup))
                .grouping(Collectors.toSet());
        lvb.checkEachBy(adGroup -> validateAdGroupContentTypeInRequest(adGroup,
                calcContentPromotionRequestAdGroupTypesByCid.getOrDefault(adGroup.getCampaignId(), emptySet())),
                When.valueIs(adGroup -> CONTENT_PROMOTION_ADGROUPS_TYPES.contains(adGroup.getType())));

        Map<Integer, Long> campaignIdByAdGroupIndex = EntryStream.of(adGroups)
                .filterValues(t -> CONTENT_PROMOTION_ADGROUPS_TYPES.contains(t.getType()))
                .mapValues(AdGroup::getCampaignId)
                .toMap();

        //набор разрешённых группе типов контента на основе существующих в базе по индексу группы
        Map<Integer, HashSet<ContentPromotionAdgroupType>> allowedTypesPromoByIndex =
                calculateAllowedContentPromotionTypes(campaignIdByAdGroupIndex);

        lvb.checkEachBy((index, adGroup) -> validateAdGroupContentTypeWithExisting(adGroup,
                allowedTypesPromoByIndex.get(index)),
                When.isValidAnd(When.valueIs(adGroup -> CONTENT_PROMOTION_ADGROUPS_TYPES.contains(adGroup.getType()))));

        return lvb.getResult();
    }

    private static ContentPromotionAdgroupType calcContentPromotionAdgroupType(AdGroup adGroup) {
        if (adGroup.getType() == AdGroupType.CONTENT_PROMOTION_VIDEO) {
            return ContentPromotionAdgroupType.VIDEO;
        } else {
            return ((ContentPromotionAdGroup) adGroup).getContentPromotionType();
        }
    }

    //Вычисление набора разрешённых группе типов контента на основе существующих в базе по индексу группы
    private Map<Integer, HashSet<ContentPromotionAdgroupType>> calculateAllowedContentPromotionTypes(
            Map<Integer, Long> campaignIdByRequestAdGroupIndex) {
        if (campaignIdByRequestAdGroupIndex.isEmpty()) {
            return new HashMap<>();
        }
        Set<Long> campaignIds = campaignIdByRequestAdGroupIndex.values().stream().collect(Collectors.toSet());
        Map<Long, List<Long>> adGroupIdsByCampaignIds = adGroupRepository.getAdGroupIdsByCampaignIds(shard,
                campaignIds);
        Set<Long> adGroupIdsInDb = StreamEx.of(adGroupIdsByCampaignIds.values())
                .flatCollection(Function.identity())
                .toSet();
        //список групп данных кампаний в базе
        List<AdGroup> dbAdGroups = adGroupRepository.getAdGroups(shard, adGroupIdsInDb);
        Map<Long, List<ContentPromotionAdgroupType>> contentPromotionDbAdgroupTypesByCid = StreamEx.of(dbAdGroups)
                .mapToEntry(AdGroup::getCampaignId,
                        ContentPromotionContentTypesValidator::calcContentPromotionAdgroupType)
                .grouping();

        return EntryStream.of(campaignIdByRequestAdGroupIndex)
                .mapValues(cid -> contentPromotionDbAdgroupTypesByCid.getOrDefault(cid,
                        allAllowableContentPromotionTypeValues()))
                .mapValues(HashSet::new)
                .toMap();
    }

    private static List<ContentPromotionAdgroupType> allAllowableContentPromotionTypeValues() {
        return StreamEx.of(ContentPromotionAdgroupType.values()).toList();
    }

    private ValidationResult<AdGroup, Defect> validateAdGroupContentTypeWithExisting(
            AdGroup adGroup, Set<ContentPromotionAdgroupType> allowedTypes) {
        ContentPromotionAdgroupType type = calcContentPromotionAdgroupType(adGroup);
        //type == null отлавливается в ContentPromotionAdGroupValidation
        return type == null || allowedTypes.contains(calcContentPromotionAdgroupType(adGroup)) ?
                ValidationResult.success(adGroup) :
                ValidationResult.failed(adGroup, contentPromotionDistinctTypesWithExisting());
    }

    private ValidationResult<AdGroup, Defect> validateAdGroupContentTypeInRequest(
            AdGroup adGroup, @Nonnull Set<ContentPromotionAdgroupType> typesInRequest) {
        return typesInRequest.size() > 1 ? ValidationResult.failed(adGroup, contentPromotionSeveralTypesNotAllowed()) :
                ValidationResult.success(adGroup);
    }
}
