package ru.yandex.direct.core.entity.adgroup.service.validation;

import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.adgroup.service.bstags.AdGroupBsTagsSettings;
import ru.yandex.direct.core.entity.adgroup.service.bstags.AllowedBsTagsValidator;
import ru.yandex.direct.core.entity.adgroup.service.geotree.AdGroupGeoTreeProvider;
import ru.yandex.direct.core.entity.adgroup.service.validation.types.AdGroupTypeSpecificValidationProvider;
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeSource;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.AccessDefectPresets;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignAccessDefects;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessConstraint;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignAccessType;
import ru.yandex.direct.core.entity.client.repository.ClientRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.hypergeo.model.HyperGeo;
import ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseValidator;
import ru.yandex.direct.core.entity.minuskeywordspack.repository.MinusKeywordsPackRepository;
import ru.yandex.direct.core.entity.tag.repository.TagRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.queryrec.model.Language;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.util.ModelChangesValidationTool;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupConstraints.hasValidTypeForUpdate;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.adGroupTypeNotSupported;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.duplicatedObject;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseBeforeNormalizationValidator.minusKeywordsAreValidBeforeNormalization;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseConstraints.GROUP_MINUS_KEYWORDS_MAX_LENGTH_BEFORE_NORMALIZATION;
import static ru.yandex.direct.core.validation.defects.RightsDefects.noRights;
import static ru.yandex.direct.feature.FeatureName.HYPERLOCAL_GEO_FOR_UC_CAMPAIGNS_ENABLED_FOR_DNA;
import static ru.yandex.direct.feature.FeatureName.HYPERLOCAL_GEO_IN_SMART_AND_DO_FOR_DNA;
import static ru.yandex.direct.utils.CommonUtils.memoize;
import static ru.yandex.direct.utils.FunctionalUtils.nullSafetyFlatMap;
import static ru.yandex.direct.validation.defect.CommonDefects.objectNotFound;
import static ru.yandex.direct.validation.result.ValidationResult.getValidItems;
import static ru.yandex.direct.validation.result.ValidationResult.replaceModelListByModelChangesList;

@Service
@ParametersAreNonnullByDefault
public class UpdateAdGroupValidationService {
    public static final int MAX_ELEMENTS_PER_OPERATION = 1000;

    private static final CampaignAccessDefects CAMPAIGN_ACCESS_DEFECTS =
            AccessDefectPresets.DEFAULT_DEFECTS.toBuilder()
                    .withTypeNotAllowable(objectId -> objectNotFound())
                    .withNotVisible(objectId -> objectNotFound())
                    .withTypeNotSupported(adGroupId -> adGroupTypeNotSupported())
                    .withNoRights(objectId -> noRights())
                    .build();

    private final CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory;
    private final AdGroupValidationService commonValidationService;
    private final AdGroupTypeSpecificValidationProvider typeSpecificValidationProvider;
    private final AdGroupRepository adGroupRepository;
    private final TagRepository tagRepository;
    private final MinusKeywordsPackRepository packRepository;
    private final ModelChangesValidationTool preValidationTool;
    private final AdGroupLanguageGeoValidator languageGeoValidator;
    private final FeatureService featureService;
    private final ClientRepository clientRepository;

    @Autowired
    public UpdateAdGroupValidationService(
            CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory,
            AdGroupValidationService commonValidationService,
            AdGroupTypeSpecificValidationProvider typeSpecificValidationProvider,
            AdGroupRepository adGroupRepository,
            TagRepository tagRepository,
            MinusKeywordsPackRepository packRepository,
            AdGroupLanguageGeoValidator languageGeoValidator, FeatureService featureService,
            ClientRepository clientRepository) {
        this.campaignSubObjectAccessCheckerFactory = campaignSubObjectAccessCheckerFactory;
        this.commonValidationService = commonValidationService;
        this.typeSpecificValidationProvider = typeSpecificValidationProvider;
        this.adGroupRepository = adGroupRepository;
        this.tagRepository = tagRepository;
        this.packRepository = packRepository;
        this.featureService = featureService;
        this.preValidationTool = ModelChangesValidationTool.builder()
                .minSize(1).maxSize(MAX_ELEMENTS_PER_OPERATION)
                .duplicatedItemDefect(duplicatedObject())
                .build();
        this.languageGeoValidator = languageGeoValidator;
        this.clientRepository = clientRepository;
    }

    public ValidationResult<List<ModelChanges<AdGroup>>, Defect> preValidateAccess(
            List<ModelChanges<AdGroup>> modelChangesList,
            Long operatorUid,
            ClientId clientId,
            int shard) {
        Set<Long> adGroupIds = modelChangesList.stream()
                .map(ModelChanges::getId)
                .collect(toSet());
        Set<Long> clientExistingAdGroupIds = adGroupRepository.getClientExistingAdGroupIds(shard, clientId, adGroupIds);

        ValidationResult<List<ModelChanges<AdGroup>>, Defect> preValidationResult =
                new ValidationResult<>(modelChangesList);
        preValidationTool.validateModelChangesList(preValidationResult, () -> clientExistingAdGroupIds);
        // ошибка на самом списке элементов - нет смысла проверять элементы
        if (preValidationResult.hasErrors()) {
            return preValidationResult;
        }

        CampaignSubObjectAccessConstraint campaignAccessConstraint = campaignSubObjectAccessCheckerFactory
                .newAdGroupChecker(operatorUid, clientId, clientExistingAdGroupIds)
                .createValidator(CampaignAccessType.READ_WRITE, CAMPAIGN_ACCESS_DEFECTS)
                .getAccessConstraint();
        return new ListValidationBuilder<>(preValidationResult)
                .checkEach(
                        (Constraint<ModelChanges<AdGroup>, Defect>) mc -> campaignAccessConstraint
                                .apply(mc.getId()),
                        When.isValid())
                .getResult();
    }

    /**
     * Провалидировать изменения в группах объявлений перед загрузкой моделей.
     * В preValidationResult передавать результат preValidateAccess.
     */
    public ValidationResult<List<ModelChanges<AdGroup>>, Defect> preValidate(
            ValidationResult<List<ModelChanges<AdGroup>>, Defect> preValidationResult,
            Map<Long, HyperGeo> hyperGeoById,
            Map<Long, CampaignTypeSource> campaignTypeSourceById,
            AdGroupGeoTreeProvider geoTreeProvider,
            ClientId clientId,
            int shard) {

        // На данный момент все дублирующиеся элементы уже помечены как невалидные - для них нет смысла
        // подготавливать isPropertyChanged
        Map<Long, ModelChanges<AdGroup>> validIdToModelChanges = StreamEx.of(getValidItems(preValidationResult))
                .mapToEntry(ModelChanges::getId, Function.identity())
                .toMap();
        BiFunction<AdGroup, ModelProperty, Boolean> isPropertyChanged =
                (adGroup, modelProperty) -> Optional.ofNullable(validIdToModelChanges.get(adGroup.getId()))
                        .map(mc -> mc.isPropChanged(modelProperty))
                        .orElse(false);

        Set<Long> validAdGroupIds = validIdToModelChanges.keySet();
        List<AdGroup> allAdGroups = preValidationResult.getValue().stream()
                .map(ModelChanges::toModel)
                .collect(toList());
        List<AdGroup> validAdGroups = allAdGroups.stream()
                .filter(adGroup -> validAdGroupIds.contains(adGroup.getId()))
                .collect(toList());
        Map<Long, Long> campaignIdsByAdGroupIds = adGroupRepository.getCampaignIdsByAdGroupIds(shard, validAdGroupIds);
        validAdGroups.forEach(adGroup -> {
            Long adGroupId = adGroup.getId();
            adGroup.setCampaignId(campaignIdsByAdGroupIds.get(adGroupId));
        });

        Set<Long> allTags = nullSafetyFlatMap(validAdGroups, AdGroup::getTags, toSet());
        Map<Long, List<Long>> existingCampaignsTags = tagRepository.getCampaignsTagIds(shard, clientId, allTags);

        Set<Long> allLibPackIds = nullSafetyFlatMap(validAdGroups, AdGroup::getLibraryMinusKeywordsIds, toSet());
        Set<Long> existingLibPackIds =
                packRepository.getClientExistingLibraryMinusKeywordsPackIds(shard, clientId, allLibPackIds);

        boolean hyperlocalGeoForUcCampaigns = featureService
                .isEnabledForClientId(clientId, HYPERLOCAL_GEO_FOR_UC_CAMPAIGNS_ENABLED_FOR_DNA);
        boolean hyperGeoForSmartAndDynamicCampaigns = featureService
                .isEnabledForClientId(clientId, HYPERLOCAL_GEO_IN_SMART_AND_DO_FOR_DNA);

        ValidationResult<List<AdGroup>, Defect> simpleValidationResult =
                ListValidationBuilder.of(allAdGroups, Defect.class)
                        .checkEachBy(adGroup -> commonValidationService.validateAdGroup(
                                        adGroup,
                                        campaignTypeSourceById,
                                        hyperlocalGeoForUcCampaigns,
                                        hyperGeoForSmartAndDynamicCampaigns,
                                        (hyperlocalIsEnabled, multipleSegmentsInHyperGeo) -> preValidateAdGroup(
                                                adGroup,
                                                geoTreeProvider.getGeoTree(adGroup),
                                                isPropertyChanged,
                                                existingCampaignsTags,
                                                existingLibPackIds,
                                                hyperGeoById,
                                                hyperlocalIsEnabled,
                                                multipleSegmentsInHyperGeo)
                                ), When.isValid()
                        )
                        .getResult();

        preValidationResult.merge(replaceModelListByModelChangesList(
                simpleValidationResult, preValidationResult.getValue()));

        return new ListValidationBuilder<>(preValidationResult)
                .checkBy(list -> typeSpecificValidationProvider.validateModelChanges(clientId, list))
                .getResult();
    }

    public Validator<ModelChanges<AdGroup>, Defect> minusKeywordsBeforeNormalizationValidator(
            MinusPhraseValidator.ValidationMode minusPhraseValidationMode) {
        return modelChanges -> {
            ItemValidationBuilder<ModelChanges<AdGroup>, Defect> vb =
                    ItemValidationBuilder.of(modelChanges, Defect.class);

            if (modelChanges.isPropChanged(AdGroup.MINUS_KEYWORDS)) {
                vb.item(modelChanges.getChangedProp(AdGroup.MINUS_KEYWORDS), AdGroup.MINUS_KEYWORDS.name())
                        .checkBy(minusKeywordsAreValidBeforeNormalization(
                                GROUP_MINUS_KEYWORDS_MAX_LENGTH_BEFORE_NORMALIZATION,
                                minusPhraseValidationMode));
            }

            return vb.getResult();
        };
    }

    private ValidationResult<AdGroup, Defect> preValidateAdGroup(
            AdGroup adGroup,
            GeoTree geoTree,
            BiFunction<AdGroup, ModelProperty, Boolean> isPropertyChanged,
            Map<Long, List<Long>> campaignsTags,
            Set<Long> existingLibPackIds,
            Map<Long, HyperGeo> hyperGeoById,
            boolean hyperlocalIsEnabled,
            boolean multipleSegmentsInHyperGeo
    ) {
        return ModelItemValidationBuilder.of(adGroup)
                .checkBy(ag -> commonValidationService
                        .validateAdGroup(ag, geoTree, isPropertyChanged, campaignsTags, existingLibPackIds,
                                hyperGeoById, hyperlocalIsEnabled, multipleSegmentsInHyperGeo))
                .getResult();
    }

    /**
     * Провалидировать тип группы объявлений после загрузки моделей, но перед применением изменений
     */
    public ValidationResult<List<ModelChanges<AdGroup>>, Defect> validateAdGroupsType(
            ValidationResult<List<ModelChanges<AdGroup>>, Defect> preValidateResult,
            Map<Long, AdGroup> models) {
        return new ListValidationBuilder<>(preValidateResult)
                .checkEach(hasValidTypeForUpdate(models), When.isValid())
                .getResult();
    }

    /**
     * Провалидировать группы объявлений после загрузки моделей и применения изменений
     */
    public ValidationResult<List<AdGroup>, Defect> validate(
            int shard, ClientId clientId, Long operatorUid, ValidationResult<List<AdGroup>, Defect> preValidationResult,
            Map<Integer, AppliedChanges<AdGroup>> appliedChangesForValidModelChanges,
            Map<AdGroup, AdGroupBsTagsSettings> adGroupsBsTagsSettings, Set<Long> adGroupsWithChangedGeoIds,
            boolean validateInterconnections) {

        Objects.requireNonNull(preValidationResult, "preValidationResult");
        Objects.requireNonNull(adGroupsWithChangedGeoIds, "adGroupsWithChangedGeoIds");

        var validAdGroupsAppliedChanges = EntryStream.of(appliedChangesForValidModelChanges)
                .mapToKey((i, appliedChanges) -> appliedChanges.getModel())
                .toCustomMap(IdentityHashMap::new);

        // Пытаемся по-максимуму оттянуть момент обращения к базе
        Supplier<Map<Long, Language>> adGroupsLangByCampaign = memoize(
                () -> EntryStream.of(adGroupRepository.getAdGroupsLangFromCampaign(shard, adGroupsWithChangedGeoIds))
                        .mapValues(Language::getByName)
                        .toMap());

        var allowedBsTagsValidator = new AllowedBsTagsValidator(adGroupsBsTagsSettings, featureService, clientId,
                operatorUid);

        Long clientRegionId = clientRepository.getCountryRegionIdByClientId(shard, clientId).orElse(null);

        Constraint<AdGroup, Defect> languageGeoConstraint = languageGeoValidator.createConstraint(
                adGroupsWithChangedGeoIds,
                adGroupId -> adGroupsLangByCampaign.get().get(adGroupId), clientId, clientRegionId);

        return new ListValidationBuilder<>(preValidationResult)
                .checkBy(list -> typeSpecificValidationProvider.validateAdGroups(clientId, list))
                .checkEachBy(allowedBsTagsValidator,
                        When.isValidAnd(When.valueIs(adGroup -> {
                            var adGroupChanges = validAdGroupsAppliedChanges.get(adGroup);
                            return adGroupChanges.changed(AdGroup.PAGE_GROUP_TAGS)
                                    || adGroupChanges.changed(AdGroup.TARGET_TAGS);
                        })))
                .checkEachBy(adGroup -> validateAdGroup(adGroup, validateInterconnections, languageGeoConstraint),
                        When.isValid())
                .getResult();
    }

    private ValidationResult<AdGroup, Defect> validateAdGroup(
            AdGroup adGroup, boolean validateInterconnections,
            Constraint<AdGroup, Defect> languageGeoConstraint) {
        ModelItemValidationBuilder<AdGroup> vb = ModelItemValidationBuilder.of(adGroup);

        vb.item(AdGroup.GEO)
                .check(geoIds -> languageGeoConstraint.apply(adGroup), When.isTrue(validateInterconnections));

        return vb.getResult();
    }
}
