package ru.yandex.direct.core.entity.adgroup.service.validation.types;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Predicate;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.PageBlock;
import ru.yandex.direct.core.entity.placements.model1.Placement;
import ru.yandex.direct.core.entity.placements.model1.PlacementBlock;
import ru.yandex.direct.core.entity.placements.model1.PlacementType;
import ru.yandex.direct.core.entity.placements.repository.PlacementRepository;
import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.maxListSize;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.defect.CommonDefects.objectNotFound;

/**
 * Из-за того, что при генерации моделей создаются вспомогательные интерфейсы-холдеры,
 * пришлось добавить второй тип-аргумент в шаблон и местами добавить явные касты.
 *
 * @param <T>           Тип конкретной группы (которая implements TPropHolder и содержит проперти)
 * @param <TPropHolder> Тип интерфейса-холдера для проперти
 */
@ParametersAreNonnullByDefault
public class AdGroupPageBlocksValidator<T extends AdGroup, TPropHolder extends Model> implements Validator<T, Defect> {

    private final PlacementRepository placementRepository;

    private final ModelProperty<TPropHolder, List<PageBlock>> pageBlocksProperty;

    private final Map<Long, Placement> existingPlacements;

    private final PlacementType allowedPlacementType;

    private final int maxCount;

    private final boolean testingPagesFeatureEnabled;

    public AdGroupPageBlocksValidator(PlacementRepository placementRepository, List<T> adGroups,
                                      ModelProperty<TPropHolder, List<PageBlock>> pageBlocksProperty,
                                      PlacementType allowedPlacementType, int maxCount,
                                      boolean testingPagesFeatureEnabled) {
        this.placementRepository = placementRepository;
        this.pageBlocksProperty = pageBlocksProperty;
        this.existingPlacements = getExistingPlacements(adGroups, pageBlocksProperty);
        this.allowedPlacementType = allowedPlacementType;
        this.maxCount = maxCount;
        this.testingPagesFeatureEnabled = testingPagesFeatureEnabled;
    }

    @SuppressWarnings("unchecked")
    private Map<Long, Placement> getExistingPlacements(List<T> adGroups,
                                                       ModelProperty<TPropHolder, List<PageBlock>> pageBlocksProperty) {
        Set<Long> allAdGroupPlacementIds = StreamEx.of(adGroups)
                .filter(Objects::nonNull)
                .flatCollection((T model) -> pageBlocksProperty.get((TPropHolder) model))
                .filter(Objects::nonNull)
                .map(PageBlock::getPageId)
                .filter(Objects::nonNull)
                .toSet();
        return placementRepository.getPlacements(allAdGroupPlacementIds);
    }

    @Override
    @SuppressWarnings("unchecked")
    public ValidationResult<T, Defect> apply(T adGroup) {
        ModelItemValidationBuilder<T> vb = ModelItemValidationBuilder.of(adGroup);
        vb.list((ModelProperty<T, List<PageBlock>>) pageBlocksProperty)
                .check(notNull())
                .check(notEmptyCollection())
                .check(maxListSize(maxCount));

        if (!vb.list((ModelProperty<T, List<PageBlock>>) pageBlocksProperty).getResult().hasAnyErrors()) {
            vb.list((ModelProperty<T, List<PageBlock>>) pageBlocksProperty)
                    .checkEach(unique(), When.isValid())
                    .checkEach(notNull(), When.isValid())
                    .checkEachBy(this::validatePageBlock, When.isValid());
        }

        return vb.getResult();
    }

    private ValidationResult<PageBlock, Defect> validatePageBlock(PageBlock pageBlock) {
        Long pageId = pageBlock.getPageId();
        Placement<?> placement = pageId == null ? null : existingPlacements.get(pageId);

        ItemValidationBuilder<PageBlock, Defect> vb = ItemValidationBuilder.of(pageBlock);

        vb.item(pageBlock.getPageId(), "pageId")
                .check(notNull())
                .check(pageIdExists(placement))
                .check(pageTypeIsAllowed(placement))
                .check(clientCanTargetOnTestingPage(), When.isTrue(placement != null && placement.isTesting()));

        if (!vb.getResult().hasAnyErrors()) {
            vb.item(pageBlock.getImpId(), "impId")
                    .check(notNull())
                    .check(blockIdExists(placement));
        }

        return vb.getResult();
    }

    private Constraint<Long, Defect> pageIdExists(@Nullable Placement<?> placement) {
        return Constraint.fromPredicate(pageId -> placement != null, objectNotFound());
    }

    private Constraint<Long, Defect> pageTypeIsAllowed(@Nullable Placement<?> placement) {
        Predicate<Long> predicate = pageId ->
                placement == null || placement.getType() == allowedPlacementType;
        return Constraint.fromPredicate(predicate, objectNotFound());
    }

    private Constraint<Long, Defect> clientCanTargetOnTestingPage() {
        return Constraint.fromPredicate(pageId -> testingPagesFeatureEnabled, objectNotFound());
    }

    private Constraint<Long, Defect> blockIdExists(@Nullable Placement<?> placement) {
        Predicate<Long> predicate = blockId -> {
            Map<Long, PlacementBlock> blocks = placement == null ?
                    null : listToMap(placement.getBlocks(), PlacementBlock::getBlockId);
            return blocks == null || blocks.containsKey(blockId);
        };
        return Constraint.fromPredicate(predicate, objectNotFound());
    }
}
