package ru.yandex.direct.core.entity.adgroup.service.validation.types;

import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupPriceSales;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupCpmPriceUtils;
import ru.yandex.direct.core.entity.campaign.model.CpmPriceCampaign;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.retargeting.model.Rule;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.DefaultValidator;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.emptyList;
import static ru.yandex.direct.core.entity.adgroup.service.AdGroupCpmPriceUtils.isDefaultPriority;
import static ru.yandex.direct.core.entity.adgroup.service.AdGroupCpmPriceUtils.isSpecificPriority;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.cpmPriceInvalidPriority;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.onlyOneDefaultAdGroupAllowed;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.priceSalesDisallowedAdGroupTypes;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

@ParametersAreNonnullByDefault
public class AdGroupPriceSalesAddValidator<T extends AdGroupPriceSales> implements DefaultValidator<T> {

    private final Map<Long, Long> existingAdGroupsPriority;
    private final Map<Long, CpmPriceCampaign> campaigns;
    private final Map<Long, List<Long>> campaignsExistingAdGroupIds;
    private final Map<Long, Set<T>> campaignsNewAdGroups;
    private final GeoTree priceSalesGeoTree;
    private final Map<Long, PricePackage> pricePackages;
    private final Map<Long, AdGroup> defAdGroupByCampaignId;

    AdGroupPriceSalesAddValidator(
            Map<Long, Long> existingAdGroupsPriority,
            Map<Long, CpmPriceCampaign> campaigns,
            Map<Long, List<Long>> campaignsExistingAdGroupIds,
            Map<Long, Set<T>> campaignsNewAdGroups,
            GeoTree priceSalesGeoTree,
            Map<Long, PricePackage> pricePackage,
            Map<Long, AdGroup> defAdGroupByCampaignId) {
        this.existingAdGroupsPriority = existingAdGroupsPriority;
        this.campaigns = campaigns;
        this.campaignsExistingAdGroupIds = campaignsExistingAdGroupIds;
        this.campaignsNewAdGroups = campaignsNewAdGroups;
        this.priceSalesGeoTree = priceSalesGeoTree;
        this.pricePackages = pricePackage;
        this.defAdGroupByCampaignId = defAdGroupByCampaignId;
    }

    @Override
    public ValidationResult<T, Defect> apply(T adGroup) {
        var priority = adGroup.getPriority();

        var vb = ModelItemValidationBuilder.of(adGroup);
        var pricePackage= pricePackages.get(campaigns.get(adGroup.getCampaignId()).getPricePackageId());

        //Если на пакете не нужна дефолтная группа, а ее пытаются сохранить, значит что-то тут не так,
        // бросаем ошибку AdGroupDefectIds.Gen.PRICE_SALES_INVALID_PRIORITY
        vb.check(fromPredicate(adGrp -> !(!pricePackage.needDefaultAdGroup() && isDefaultPriority(priority)),
                cpmPriceInvalidPriority()));
        vb.checkBy(this::validateDefaultAdGroup, When.isTrue(isDefaultPriority(priority)));
        vb.checkBy(this::validateSpecificAdGroup, When.isTrue(isSpecificPriority(priority)));
        vb.checkBy(this::validateAdGroupType, When.isValid());
        var categoriesValidationData = new PriceSalesAdGroupContentCategoriesValidationData(
                pricePackage, priority,
                getDefaultAdGroupContentCategoriesRulesByCampaignIds(adGroup.getCampaignId()));
        vb.list(AdGroupPriceSales.CONTENT_CATEGORIES_RETARGETING_CONDITION_RULES)
                .checkBy(new PriceSalesAdGroupContentCategoriesValidator(categoriesValidationData), When.isValid());

        return vb.getResult();
    }

    private List<Rule> getDefaultAdGroupContentCategoriesRulesByCampaignIds(Long cid) {
        var defAdGroup = defAdGroupByCampaignId.get(cid);
        if (defAdGroup == null) {
            return null;
        }
        return defAdGroup.getContentCategoriesRetargetingConditionRules();
    }

    private ValidationResult<T, Defect> validateDefaultAdGroup(T  adGroup) {
        Long campaignId = adGroup.getCampaignId();
        var campaign = campaigns.get(campaignId);
        checkNotNull(campaign, "Not found cpm price campaign with id %s", campaignId);
        var pricePackage = pricePackages.get(campaign.getPricePackageId());

        var vb = ModelItemValidationBuilder.of(adGroup);

        vb.item(AdGroupPriceSales.GEO)
                .checkBy(new PriceSalesDefaultAdGroupGeoValidator(priceSalesGeoTree, campaign, pricePackage));

        vb.item(AdGroupPriceSales.PRIORITY)
                .check(campaignHasNoDefaultAdGroup(campaignId))
                .check(addingExactlyOneDefaultAdGroup(campaignId));

        return vb.getResult();
    }

    private Constraint<Long, Defect> campaignHasNoDefaultAdGroup(Long campaignId) {
        var existingAdGroupIdsInCampaign = campaignsExistingAdGroupIds.getOrDefault(campaignId, emptyList());
        var alreadyHasDefaultAdGroup = existingAdGroupIdsInCampaign.stream()
                .map(existingAdGroupsPriority::get)
                .anyMatch(AdGroupCpmPriceUtils::isDefaultPriority);
        return priority -> alreadyHasDefaultAdGroup ? onlyOneDefaultAdGroupAllowed() : null;
    }

    private Constraint<Long, Defect> addingExactlyOneDefaultAdGroup(Long campaignId) {
        var newAdGroupsInCampaign = campaignsNewAdGroups.get(campaignId);
        var newDefaultAdGroupsCount = newAdGroupsInCampaign.stream()
                .map(AdGroupPriceSales::getPriority)
                .filter(AdGroupCpmPriceUtils::isDefaultPriority)
                .count();
        return priority -> newDefaultAdGroupsCount > 1 ? onlyOneDefaultAdGroupAllowed() : null;
    }

    private ValidationResult<T, Defect> validateSpecificAdGroup(T adGroup) {
        Long campaignId = adGroup.getCampaignId();
        var campaign = campaigns.get(campaignId);
        checkNotNull(campaign, "Not found cpm price campaign with id %s", campaignId);
        var pricePackage = pricePackages.get(campaign.getPricePackageId());

        var vb = ModelItemValidationBuilder.of(adGroup);

        vb.item(AdGroupPriceSales.GEO)
                .checkBy(new PriceSalesSpecificAdGroupGeoValidator(priceSalesGeoTree, campaign, pricePackage,
                        defAdGroupByCampaignId.get(campaignId)));

        return vb.getResult();
    }

    /**
     * Проверка типа группы, что он разрешен пакетом
     *
     * @param availableAdGroupTypes список разрешенных типов группы для в пакете
     */
    public static Constraint<AdGroupType, Defect> isAllowedAdGroupType(
            Set<AdGroupType> availableAdGroupTypes
    ) {
        return fromPredicate(adGroupType -> availableAdGroupTypes != null && availableAdGroupTypes.contains(adGroupType),
                priceSalesDisallowedAdGroupTypes());
    }

    private ValidationResult<T, Defect> validateAdGroupType(T adGroup) {
        var vb = ModelItemValidationBuilder.of(adGroup);
        var campaignId = adGroup.getCampaignId();
        var campaign = campaigns.get(campaignId);

        PricePackage pricePackage = pricePackages.get(campaign.getPricePackageId());
        vb.item(AdGroupPriceSales.TYPE)
                .check(isAllowedAdGroupType(pricePackage.getAvailableAdGroupTypes()));

        return vb.getResult();
    }
}
