package ru.yandex.direct.core.entity.adgroup.service.validation.types;

import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupPriceSales;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithType;
import ru.yandex.direct.core.entity.campaign.model.CpmPriceCampaign;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.retargeting.model.Rule;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelChangesValidationBuilder;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.core.entity.adgroup.service.AdGroupCpmPriceUtils.isDefaultPriority;
import static ru.yandex.direct.core.entity.adgroup.service.AdGroupCpmPriceUtils.isSpecificPriority;
import static ru.yandex.direct.core.validation.defects.RightsDefects.forbiddenToChange;
import static ru.yandex.direct.validation.constraint.CommonConstraints.unconditional;

/**
 * Гео проверяем в {@link AdGroupPriceSalesModelChangesValidator} и {@link AdGroupPriceSalesAddValidator},
 * а не в {@link AdGroupPriceSalesValidator}.
 * Сделано так, потому что при обновлении клиентский гео преобразуется после превалидации,
 * т.о. доступ к клиентскому гео есть только на этапе превалидации.
 */
@ParametersAreNonnullByDefault
public class AdGroupPriceSalesModelChangesValidator<T extends AdGroupPriceSales>  implements
        Validator<ModelChanges<T>, Defect> {

    private final Map<Long, Long> existingAdGroupsPriority;
    private final Map<Long, CampaignWithType> campaignWithTypeByAdGroupId;
    private final Map<Long, CpmPriceCampaign> campaigns;
    private final GeoTree priceSalesGeoTree;
    private final Map<Long, PricePackage> pricePackages;
    private final Map<Long, AdGroup> defAdGroupByCampaignId;

    public AdGroupPriceSalesModelChangesValidator(
            Map<Long, Long> existingAdGroupsPriority,
            Map<Long, CampaignWithType> campaignWithTypeByAdGroupId,
            Map<Long, CpmPriceCampaign> campaigns,
            GeoTree priceSalesGeoTree,
            Map<Long, PricePackage> pricePackages,
            Map<Long, AdGroup> defAdGroupByCampaignId) {
        this.existingAdGroupsPriority = existingAdGroupsPriority;
        this.campaignWithTypeByAdGroupId = campaignWithTypeByAdGroupId;
        this.campaigns = campaigns;
        this.priceSalesGeoTree = priceSalesGeoTree;
        this.pricePackages = pricePackages;
        this.defAdGroupByCampaignId = defAdGroupByCampaignId;
    }

    @Override
    public ValidationResult<ModelChanges<T>, Defect> apply(
            ModelChanges<T> changes) {
        var priority = existingAdGroupsPriority.get(changes.getId());
        checkNotNull(priority);
        var campaign = getCampaign(changes);
        var pricePackage = pricePackages.get(campaign.getPricePackageId());

        var vb = ModelChangesValidationBuilder.of(changes);

        vb.item(AdGroupPriceSales.PRIORITY)
                .check(unconditional(forbiddenToChange()));

        vb.checkBy(this::validateDefaultAdGroupChanges, When.isTrue(isDefaultPriority(priority)));
        vb.checkBy(this::validateSpecificAdGroupChanges, When.isTrue(isSpecificPriority(priority)));
        var categoriesValidationData = new PriceSalesAdGroupContentCategoriesValidationData(
                pricePackage, priority,
                getDefaultAdGroupContentCategoriesRulesByCampaignIds(campaign.getId()));
        vb.item(AdGroupPriceSales.CONTENT_CATEGORIES_RETARGETING_CONDITION_RULES)
                .checkBy(new PriceSalesAdGroupContentCategoriesValidator(categoriesValidationData), When.isValid());

        return vb.getResult();
    }

    private List<Rule> getDefaultAdGroupContentCategoriesRulesByCampaignIds(Long cid) {
        var defAdGroup = defAdGroupByCampaignId.get(cid);
        if (defAdGroup == null) {
            return null;
        }
        return defAdGroup.getContentCategoriesRetargetingConditionRules();
    }

    private ValidationResult<ModelChanges<T>, Defect> validateDefaultAdGroupChanges(
            ModelChanges<T> changes) {
        var campaign = getCampaign(changes);
        var pricePackage = pricePackages.get(campaign.getPricePackageId());
        var vb = ModelChangesValidationBuilder.of(changes);

        vb.item(AdGroupPriceSales.GEO)
                .checkBy(new PriceSalesDefaultAdGroupGeoValidator(priceSalesGeoTree, campaign, pricePackage));

        return vb.getResult();
    }

    private ValidationResult<ModelChanges<T>, Defect> validateSpecificAdGroupChanges(
            ModelChanges<T> changes) {
        var campaign = getCampaign(changes);
        var pricePackage = pricePackages.get(campaign.getPricePackageId());
        var vb = ModelChangesValidationBuilder.of(changes);

        vb.item(AdGroupPriceSales.GEO)
                .checkBy(new PriceSalesSpecificAdGroupGeoValidator(priceSalesGeoTree, campaign, pricePackage,
                        defAdGroupByCampaignId.get(campaign.getId())));

        return vb.getResult();
    }

    private CpmPriceCampaign getCampaign(ModelChanges<T> changes) {
        var adGroupId = changes.getId();

        var campaignWithType = campaignWithTypeByAdGroupId.get(adGroupId);
        checkNotNull(campaignWithType, "Not found campaign for adgroup with id %s", adGroupId);

        var campaignId = campaignWithType.getId();
        var campaign = checkNotNull(campaigns.get(campaignWithType.getId()));
        checkNotNull(campaign, "Not found cpm price campaign with id %s", campaignId);

        return campaign;
    }
}
