package ru.yandex.direct.core.entity.adgroup.service.validation.types;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Sets;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupPriceSales;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithType;
import ru.yandex.direct.core.entity.campaign.model.CpmPriceCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.pricepackage.repository.PricePackageRepository;
import ru.yandex.direct.core.entity.pricepackage.service.PricePackageService;

import static ru.yandex.direct.utils.CollectionUtils.flatToList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class AdGroupPriceSalesValidatorFactory {

    private final CampaignTypedRepository campaignTypedRepository;
    private final AdGroupRepository adGroupRepository;
    private final PricePackageService pricePackageService;
    private final PricePackageRepository pricePackageRepository;

    @Autowired
    public AdGroupPriceSalesValidatorFactory(CampaignTypedRepository campaignTypedRepository,
                                             AdGroupRepository adGroupRepository,
                                             PricePackageService pricePackageService,
                                             PricePackageRepository pricePackageRepository) {
        this.campaignTypedRepository = campaignTypedRepository;
        this.adGroupRepository = adGroupRepository;
        this.pricePackageService = pricePackageService;
        this.pricePackageRepository = pricePackageRepository;
    }

    public AdGroupPriceSalesValidator<AdGroupPriceSales> createBeanValidator() {
        return new AdGroupPriceSalesValidator<>();
    }

    public <T extends AdGroupPriceSales> AdGroupPriceSalesAddValidator<T> createAddValidator(
            int shard, Map<Long, CampaignType> campaignsType, List<T> adGroups) {
        Map<Long, CpmPriceCampaign> cpmPriceCampaigns = campaignTypedRepository.getTypedCampaignsMap(shard,
                campaignsType, CampaignType.CPM_PRICE);
        var cpmPriceCampaignIds = cpmPriceCampaigns.keySet();

        var campaignsExistingAdGroupIds = adGroupRepository.getAdGroupIdsByCampaignIds(shard, cpmPriceCampaignIds);

        var existingAdGroupIds = flatToList(campaignsExistingAdGroupIds.values());
        var existingAdGroupsPriority = adGroupRepository.getAdGroupsPriority(shard, existingAdGroupIds);

        var campaignsNewAdGroups = StreamEx.of(adGroups)
                .groupingTo(AdGroupPriceSales::getCampaignId, Sets::newIdentityHashSet);

        return new AdGroupPriceSalesAddValidator<>(
                existingAdGroupsPriority,
                cpmPriceCampaigns,
                campaignsExistingAdGroupIds,
                campaignsNewAdGroups,
                pricePackageService.getGeoTree(),
                pricePackages(cpmPriceCampaigns.values()),
                getDefaultAdGroupByCampaignIds(shard, cpmPriceCampaignIds));
    }


    public <T extends AdGroupPriceSales> AdGroupPriceSalesModelChangesValidator<T> createModelChangesValidator(
            int shard, Map<Long, CampaignWithType> campaignWithTypeByAdGroupId) {
        var adGroupIds = campaignWithTypeByAdGroupId.keySet();
        var cpmPriceCampaignIds = StreamEx.of(campaignWithTypeByAdGroupId.values())
                .filter(campaignWithType -> campaignWithType.getType() == CampaignType.CPM_PRICE)
                .map(CampaignWithType::getId)
                .toSet();

        var adGroupsPriority = adGroupRepository.getAdGroupsPriority(shard, adGroupIds);
        var cpmPriceCampaigns =
                (Map<Long, CpmPriceCampaign>) campaignTypedRepository.getTypedCampaignsMap(shard, cpmPriceCampaignIds);

        return new AdGroupPriceSalesModelChangesValidator<>(adGroupsPriority, campaignWithTypeByAdGroupId,
                cpmPriceCampaigns, pricePackageService.getGeoTree(),
                pricePackages(cpmPriceCampaigns.values()),
                getDefaultAdGroupByCampaignIds(shard, cpmPriceCampaignIds));
    }

    private Map<Long, PricePackage> pricePackages(Collection<CpmPriceCampaign> cpmPriceCampaigns) {
        var packageIds = mapList(cpmPriceCampaigns, CpmPriceCampaign::getPricePackageId);
        return pricePackageRepository.getPricePackages(packageIds);
    }

    private Map<Long, AdGroup> getDefaultAdGroupByCampaignIds(
            int shard, Set<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return Collections.emptyMap();
        }
        var defAdGroupIdByCampaignId = adGroupRepository.getDefaultPriceSalesAdGroupIdByCampaignId(shard, campaignIds);
        if (defAdGroupIdByCampaignId.isEmpty()) {
            return Collections.emptyMap();
        }
        var adGroups = adGroupRepository.getAdGroups(shard, defAdGroupIdByCampaignId.values());
        return StreamEx.of(adGroups)
                .mapToEntry(AdGroup::getCampaignId, Function.identity())
                .filterValues(Objects::nonNull)
                .toMap();
    }
}
