package ru.yandex.direct.core.entity.adgroup.service.validation.types;

import java.util.List;
import java.util.Map;
import java.util.stream.Stream;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toMap;

/**
 * Компонент, который разруливает правильную валидацию групп объявлений
 * разных типов. Разные AdGroupTypeSpecificValidationService в большинстве случаев
 * должны быть скрыты за ним, а он разрулит, какой тип каким образом
 * валидировать.
 */
@Service
public class AdGroupTypeSpecificValidationProvider {
    private final Map<Class<? extends AdGroup>, AdGroupTypeSpecificValidationService<AdGroup>> serviceMap;

    @Autowired
    public AdGroupTypeSpecificValidationProvider(
            TextAdGroupValidation textAdGroupValidation,
            MobileContentAdGroupValidation mobileContentAdGroupValidation,
            DynamicTextAdGroupValidation dynamicTextAdGroupValidation,
            DynamicFeedAdGroupValidation dynamicFeedAdGroupValidation,
            PerformanceAdGroupValidation performanceAdGroupValidation,
            CpmBannerAdGroupValidation cpmBannerAdGroupValidation,
            CpmGeoproductAdGroupValidation cpmGeoproductAdGroupValidation,
            CpmGeoPinAdGroupValidation cpmGeoPinAdGroupValidation,
            CpmAudioAdGroupValidation cpmAudioAdGroupValidation,
            CpmVideoAdGroupValidation cpmVideoAdGroupValidation,
            CpmOutdoorAdGroupValidation cpmOutdoorAdGroupValidation,
            CpmIndoorAdGroupValidation cpmIndoorAdGroupValidation,
            CpmYndxFrontpageAdGroupValidation cpmYndxFrontpageAdGroupValidation,
            ContentPromotionAdGroupValidation contentPromotionAdGroupValidation,
            ContentPromotionVideoAdGroupValidation contentPromotionVideoAdGroupValidation,
            InternalAdGroupValidation internalAdGroupValidation,
            McBannerAdGroupValidation mcBannerAdGroupValidation
    ) {
        this(Stream.of(
                upCast(textAdGroupValidation),
                upCast(mobileContentAdGroupValidation),
                upCast(dynamicTextAdGroupValidation),
                upCast(dynamicFeedAdGroupValidation),
                upCast(performanceAdGroupValidation),
                upCast(cpmBannerAdGroupValidation),
                upCast(cpmGeoproductAdGroupValidation),
                upCast(cpmGeoPinAdGroupValidation),
                upCast(cpmAudioAdGroupValidation),
                upCast(cpmVideoAdGroupValidation),
                upCast(cpmOutdoorAdGroupValidation),
                upCast(cpmIndoorAdGroupValidation),
                upCast(cpmYndxFrontpageAdGroupValidation),
                upCast(contentPromotionAdGroupValidation),
                upCast(contentPromotionVideoAdGroupValidation),
                upCast(internalAdGroupValidation),
                upCast(mcBannerAdGroupValidation))
                .collect(toMap(AdGroupTypeSpecificValidationService::getAdGroupClass, identity()))
        );
    }

    AdGroupTypeSpecificValidationProvider(
            Map<Class<? extends AdGroup>, AdGroupTypeSpecificValidationService<AdGroup>> serviceMap) {
        this.serviceMap = serviceMap;
    }

    @SuppressWarnings("unchecked")
    static <T extends AdGroup> AdGroupTypeSpecificValidationService<AdGroup> upCast(
            AdGroupTypeSpecificValidationService<T> service) {
        return (AdGroupTypeSpecificValidationService<AdGroup>) service;
    }

    /**
     * Валидирует список изменений специфических групп в соответствующих им валидаторах.
     */
    public ValidationResult<List<ModelChanges<AdGroup>>, Defect> validateModelChanges(ClientId clientId,
                                                                                      List<ModelChanges<AdGroup>> modelChanges) {
        ListValidationBuilder<ModelChanges<AdGroup>, Defect> vb = ListValidationBuilder.of(modelChanges, Defect.class);
        serviceMap.forEach((adGroupClass, specificValidationService) -> vb.checkSublistBy(
                sublist -> specificValidationService.validateModelChanges(clientId, sublist),
                When.isValidAnd(When.valueIs(changes -> changes.getModelType() == adGroupClass))
        ));
        return vb.getResult();
    }

    /**
     * Провалидировать список групп каждого типа в специфичном валидаторе.
     * <p>
     * Это может быть нужно, например, чтобы проверить, что во всех добавляемых в РМП-кампанию группах одинаковая
     * ссылка на приложение.
     * <p>
     * Для разных клиентов валидация может отличаться в зависимости от включенных для клиента фиче-флагов.
     *
     * @param clientId клиент, для которого выполняется валидация.
     * @param adGroups список групп для валидации, могут быть разных типов
     * @return объединение результатов валидации всех типо-специфичных валидаторов
     */
    public ValidationResult<List<AdGroup>, Defect> validateAdGroups(ClientId clientId, List<AdGroup> adGroups) {
        ListValidationBuilder<AdGroup, Defect> vb = ListValidationBuilder.of(adGroups, Defect.class);
        serviceMap.forEach((adGroupClass, specificValidationService) -> vb.checkSublistBy(
                sublist -> specificValidationService.validateAdGroups(clientId, sublist),
                When.isValidAnd(When.valueIs(g -> g.getClass() == adGroupClass))
        ));
        return vb.getResult();
    }

    public ValidationResult<List<AdGroup>, Defect> validateAddAdGroups(ClientId clientId, List<AdGroup> adGroups) {
        ListValidationBuilder<AdGroup, Defect> vb = ListValidationBuilder.of(adGroups, Defect.class);
        serviceMap.forEach((adGroupClass, specificValidationService) -> vb.checkSublistBy(
                sublist -> specificValidationService.validateAddAdGroups(clientId, sublist),
                When.isValidAnd(When.valueIs(g -> g.getClass() == adGroupClass))
        ));
        return vb.getResult();

    }
}
