package ru.yandex.direct.core.entity.adgroup.service.validation.types;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Supplier;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupWithFeedId;
import ru.yandex.direct.core.entity.feed.model.Feed;
import ru.yandex.direct.core.entity.feed.model.MasterSystem;
import ru.yandex.direct.core.entity.feed.model.Source;
import ru.yandex.direct.core.entity.feed.model.UpdateStatus;
import ru.yandex.direct.core.entity.feed.service.FeedService;
import ru.yandex.direct.core.entity.feed.validation.FeedDefects;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.builder.ListItemValidator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static com.google.common.base.Preconditions.checkState;
import static java.util.function.Predicate.not;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.feedNotExist;
import static ru.yandex.direct.core.entity.feed.validation.FeedDefects.feedIdIsNotValid;
import static ru.yandex.direct.core.entity.feed.validation.FeedDefects.feedIsNotSet;
import static ru.yandex.direct.utils.CommonUtils.memoize;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;

public class AdGroupWithFeedIdValidator<T extends AdGroupWithFeedId> implements ListItemValidator<T, Defect> {

    private static final Set<UpdateStatus> VALID_UPDATE_STATUSES =
            Set.of(UpdateStatus.DONE, UpdateStatus.UPDATING, UpdateStatus.OUTDATED);
    private static final Set<UpdateStatus> VALID_UPDATE_STATUSES_FOR_SITE_FEEDS =
            Set.of(UpdateStatus.NEW);
    private static final Set<UpdateStatus> VALID_UPDATE_STATUSES_FOR_SHOP_IN_SHOP_FEEDS =
            Set.of(UpdateStatus.NEW);
    private static final Set<UpdateStatus> VALID_UPDATE_STATUSES_FOR_MANUAL_FEEDS =
            Set.of(UpdateStatus.NEW);

    private final FeedService feedService;
    private final ClientId clientId;
    private final Set<Long> adGroupIds;
    private final Set<Long> srcFeedIds;
    protected final Supplier<Map<Long, Feed>> lazyFeedById;
    private final Supplier<Set<Long>> lazyExistingFeedIds;

    public AdGroupWithFeedIdValidator(Builder builder) {
        this.feedService = builder.feedService;
        this.clientId = builder.clientId;
        this.adGroupIds = builder.adGroupIds;
        this.srcFeedIds = builder.srcFeedIds;
        this.lazyFeedById = memoize(this::getFeedById);
        this.lazyExistingFeedIds = memoize(this::getExistingFeedIds);
    }

    private Map<Long, Feed> getFeedById() {
        List<Feed> feeds = feedService.getFeeds(clientId, srcFeedIds);
        return StreamEx.of(feeds)
                .mapToEntry(Feed::getId, feed -> feed)
                .toMap();
    }

    private Set<Long> getExistingFeedIds() {
        return lazyFeedById.get().keySet();
    }

    @Override
    public ValidationResult<T, Defect> validate(int index, T adGroup) {
        checkState(adGroupIds.contains(adGroup.getId()), "Unexpected AdGroup");
        ModelItemValidationBuilder<T> vb = ModelItemValidationBuilder.of(adGroup);
        vb.item(AdGroupWithFeedId.FEED_ID)
                .check(notNull(), feedIsNotSet())
                .check(validId(), feedIdIsNotValid())
                .checkByFunction(feedIdExistValidator(), When.isValid())
                .checkByFunction(feedStatusValidator(), When.isValid());
        return vb.getResult();
    }

    private Function<Long, Defect> feedStatusValidator() {
        return feedId -> Optional.ofNullable(feedId)
                .map(lazyFeedById.get()::get)
                .filter(not(this::isValidUpdateStatus))
                .map(Feed::getUpdateStatus)
                .map(FeedDefects::feedStatusWrong)
                .orElse(null);
    }

    private boolean isValidUpdateStatus(Feed feed) {
        return VALID_UPDATE_STATUSES.contains(feed.getUpdateStatus()) ||
                // Т.к. у сайтов нет категорий, то рекламные материалы по сайтам можно начинать создавать
                // ещё до завершения обхода
                (VALID_UPDATE_STATUSES_FOR_SITE_FEEDS.contains(feed.getUpdateStatus())
                        && feed.getSource() == Source.SITE) ||
                // Фиды по маркеплейсам создаются в процессе создания кампании и не успевают обновить статус
                (VALID_UPDATE_STATUSES_FOR_SHOP_IN_SHOP_FEEDS.contains(feed.getUpdateStatus())
                        && feed.getMasterSystem() == MasterSystem.SHOP_IN_SHOP) ||
                // Ручные фиды создаются в процессе создания кампании и не успевают обновить статус
                (VALID_UPDATE_STATUSES_FOR_MANUAL_FEEDS.contains(feed.getUpdateStatus())
                        && feed.getMasterSystem() == MasterSystem.MANUAL);
    }

    private Function<Long, Defect> feedIdExistValidator() {
        return feedId -> lazyExistingFeedIds.get().contains(feedId) ? null : feedNotExist(feedId);
    }

    public static class Builder<T extends AdGroupWithFeedId> {
        private final FeedService feedService;
        private final ClientId clientId;
        private final Set<Long> adGroupIds;
        private final Set<Long> srcFeedIds;

        public Builder(FeedService feedService, ClientId clientId, List<T> adGroups) {
            this.feedService = feedService;
            this.clientId = clientId;
            this.adGroupIds = listToSet(adGroups, AdGroupWithFeedId::getId);
            this.srcFeedIds = listToSet(adGroups, AdGroupWithFeedId::getFeedId);
        }

        public AdGroupWithFeedIdValidator<T> build() {
            return new AdGroupWithFeedIdValidator<>(this);
        }
    }

}
