package ru.yandex.direct.core.entity.adgroup.service.validation.types;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.ContentPromotionAdGroup;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.contentPromotionAdGroupContentTypeChanged;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.result.PathHelper.field;

/**
 * Класс с логикой валидации групп объявлений типа CONTENT_PROMOTION при операциях добавления и обновления групп
 */
@Component
@ParametersAreNonnullByDefault
public class ContentPromotionAdGroupValidation extends
        BaseAdGroupTypeSpecificValidationService<ContentPromotionAdGroup> {

    @Override
    public ValidationResult<List<ModelChanges<ContentPromotionAdGroup>>, Defect> validateModelChanges(ClientId clientId,
                                                                                                      List<ModelChanges<ContentPromotionAdGroup>> modelChanges) {
        if (isEmpty(modelChanges)) {
            return ValidationResult.success(modelChanges);
        }
        ListValidationBuilder<ModelChanges<ContentPromotionAdGroup>, Defect> lvb =
                ListValidationBuilder.of(modelChanges);

        //Проверка, что не сменился тип контента
        lvb.checkEachBy(this::validateAdGroupContentTypeUnchanged);
        return lvb.getResult();
    }

    private ValidationResult<ModelChanges<ContentPromotionAdGroup>, Defect> validateAdGroupContentTypeUnchanged(
            ModelChanges<ContentPromotionAdGroup> modelChanges) {
        ItemValidationBuilder<ModelChanges<ContentPromotionAdGroup>, Defect> vb =
                ItemValidationBuilder.of(modelChanges, Defect.class);

        if (modelChanges.isPropChanged(ContentPromotionAdGroup.CONTENT_PROMOTION_TYPE)) {
            vb.getResult()
                    .getOrCreateSubValidationResult(field(ContentPromotionAdGroup.CONTENT_PROMOTION_TYPE.name()),
                            modelChanges.getChangedProp(ContentPromotionAdGroup.CONTENT_PROMOTION_TYPE))
                    .addError(contentPromotionAdGroupContentTypeChanged());
        }
        return vb.getResult();
    }

    //Вся валидация вынесена в ContentPromotionContentTypesValidator
    @Override
    public ValidationResult<ContentPromotionAdGroup, Defect> validateAdGroup(ContentPromotionAdGroup adGroup) {
       var vb = ModelItemValidationBuilder.of(adGroup);
       vb.item(ContentPromotionAdGroup.CONTENT_PROMOTION_TYPE).check(notNull());
       return vb.getResult();
    }

    @Override
    public Class<ContentPromotionAdGroup> getAdGroupClass() {
        return ContentPromotionAdGroup.class;
    }

}
