package ru.yandex.direct.core.entity.adgroup.service.validation.types;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CpmAudioAdGroup;
import ru.yandex.direct.core.entity.adgroup.service.complex.suboperation.UsersSegmentsValidationUtil;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.adGroupTypeNotSupported;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

@Component
@ParametersAreNonnullByDefault
public class CpmAudioAdGroupValidation implements AdGroupTypeSpecificValidationService<CpmAudioAdGroup> {
    private final FeatureService featureService;
    private final ShardHelper shardHelper;
    private final CampaignRepository campaignRepository;
    private final AdGroupPriceSalesValidatorFactory priceSalesValidatorFactory;

    @Autowired
    public CpmAudioAdGroupValidation(FeatureService featureService,
            ShardHelper shardHelper,
            CampaignRepository campaignRepository,
            AdGroupPriceSalesValidatorFactory priceSalesValidatorFactory) {
        this.featureService = featureService;
        this.shardHelper = shardHelper;
        this.campaignRepository = campaignRepository;
        this.priceSalesValidatorFactory = priceSalesValidatorFactory;
    }

    @Override
    public ValidationResult<List<ModelChanges<CpmAudioAdGroup>>, Defect> validateModelChanges(ClientId clientId,
                                                  List<ModelChanges<CpmAudioAdGroup>> modelChangesList) {
        int shard = shardHelper.getShardByClientId(clientId);
        var adGroupIds = listToSet(modelChangesList, ModelChanges::getId);
        var campaignWithTypeByAdGroupId =
                campaignRepository.getCampaignsWithTypeByAdGroupIds(shard, clientId, adGroupIds);

        AdGroupPriceSalesModelChangesValidator<CpmAudioAdGroup> priceSalesValidator =
                priceSalesValidatorFactory.createModelChangesValidator(shard, campaignWithTypeByAdGroupId);

        boolean isCpmAudioDisabled = featureService.isEnabledForClientId(clientId, FeatureName.DISABLE_CPM_AUDIO);
        ListValidationBuilder<ModelChanges<CpmAudioAdGroup>, Defect> vb = ListValidationBuilder.of(modelChangesList);
        vb.checkEachBy(validateAdGroupChange(isCpmAudioDisabled))
                .checkEachBy(priceSalesValidator, When.valueIs(
                        changes -> campaignWithTypeByAdGroupId.get(changes.getId()).getType() == CampaignType.CPM_PRICE
                ));
        return vb.getResult();
    }

    private static Validator<ModelChanges<CpmAudioAdGroup>, Defect> validateAdGroupChange(boolean isCpmAudioDisabled) {
        return modelChanges -> {
            if (isCpmAudioDisabled) {
                return ValidationResult.failed(modelChanges, adGroupTypeNotSupported());
            }
            return ValidationResult.success(modelChanges);
        };
    }

    @Override
    public ValidationResult<List<CpmAudioAdGroup>, Defect> validateAdGroups(ClientId clientId,
                                                                            List<CpmAudioAdGroup> adGroups) {
        int shard = shardHelper.getShardByClientId(clientId);
        var campaignIds = listToSet(adGroups, AdGroup::getCampaignId);
        var campaignTypes = campaignRepository.getCampaignsTypeMap(shard, campaignIds);

        var priceSalesValidator = priceSalesValidatorFactory.createBeanValidator();

        boolean isCpmAudioDisabled = featureService.isEnabledForClientId(clientId, FeatureName.DISABLE_CPM_AUDIO);

        return ListValidationBuilder.<CpmAudioAdGroup, Defect>of(adGroups)
                .checkEachBy(adGroup -> (campaignTypes.get(adGroup.getCampaignId()) == CampaignType.CPM_PRICE ) ?
                        validateCpmPriceAudioAdGroup(adGroup, priceSalesValidator) :
                        validateAdGroup(adGroup, isCpmAudioDisabled)
                )
                .getResult();
    }

    private static ValidationResult<CpmAudioAdGroup, Defect> validateAdGroup(CpmAudioAdGroup adGroup,
                                                                             boolean isCpmAudioDisabled) {
        ModelItemValidationBuilder<CpmAudioAdGroup> vb = ModelItemValidationBuilder.of(adGroup);

        UsersSegmentsValidationUtil.validateAdGroupWithUsersSegments(vb, adGroup.getType(), false);

        vb.item(AdGroup.TYPE).check(fromPredicate(k -> !isCpmAudioDisabled, adGroupTypeNotSupported()));
        return vb.getResult();
    }

    private ValidationResult<CpmAudioAdGroup, Defect> validateCpmPriceAudioAdGroup(
            CpmAudioAdGroup adGroup,
            AdGroupPriceSalesValidator priceSalesValidator) {

        ModelItemValidationBuilder<CpmAudioAdGroup> vb = ModelItemValidationBuilder.of(adGroup);

        vb.checkBy(priceSalesValidator);

        return vb.getResult();
    }

    @Override
    public ValidationResult<List<CpmAudioAdGroup>, Defect> validateAddAdGroups(
            ClientId clientId,
            List<CpmAudioAdGroup> adGroups) {
        int shard = shardHelper.getShardByClientId(clientId);
        var campaignIds = listToSet(adGroups, AdGroup::getCampaignId);
        var campaignTypes = campaignRepository.getCampaignsTypeMap(shard, campaignIds);

        AdGroupPriceSalesAddValidator<CpmAudioAdGroup> priceSalesValidator =
                priceSalesValidatorFactory.createAddValidator(shard, campaignTypes, adGroups);

        return ListValidationBuilder.<CpmAudioAdGroup, Defect>of(adGroups)
                .checkEachBy(priceSalesValidator, When.valueIs(
                        adGroup -> campaignTypes.get(adGroup.getCampaignId()) == CampaignType.CPM_PRICE
                ))
                .getResult();
    }

    @Override
    public Class<CpmAudioAdGroup> getAdGroupClass() {
        return CpmAudioAdGroup.class;
    }
}
